;;; hyperspec.el --- Browse documentation from the Common Lisp HyperSpec

;; Copyright 1997 Naggum Software

;; Author: Erik Naggum <erik@naggum.no>
;; Keywords: lisp

;; This file is not part of GNU Emacs, but distributed under the same
;; conditions as GNU Emacs, and is useless without GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Kent Pitman and Xanalys Inc. have made the text of American National
;; Standard for Information Technology -- Programming Language -- Common
;; Lisp, ANSI X3.226-1994 available on the WWW, in the form of the Common
;; Lisp HyperSpec.  This package makes it convenient to peruse this
;; documentation from within Emacs.

;;; Code:

(require 'cl-lib nil t)
(require 'cl-lib "lib/cl-lib")
(require 'browse-url)                   ;you need the Emacs 20 version
(require 'thingatpt)

(defvar common-lisp-hyperspec-root
  "http://www.lispworks.com/reference/HyperSpec/"
  "The root of the Common Lisp HyperSpec URL.
If you copy the HyperSpec to your local system, set this variable to
something like \"file://usr/local/doc/HyperSpec/\".")

;;; Added variable for CLHS symbol table. See details below.
;;;
;;; 20011201 Edi Weitz

(defvar common-lisp-hyperspec-symbol-table nil
  "The HyperSpec symbol table file.
If you copy the HyperSpec to your local system, set this variable to
the location of the symbol table which is usually \"Map_Sym.txt\"
or \"Symbol-Table.text\".")

(defvar common-lisp-hyperspec-history nil
  "History of symbols looked up in the Common Lisp HyperSpec.")

(defvar common-lisp-hyperspec--symbols (make-hash-table :test 'equal)
  "Map a symbol name to its list of relative URLs.")

;; Lookup NAME in 'common-lisp-hyperspec--symbols´
(defun common-lisp-hyperspec--find (name)
  "Get the relative url of a Common Lisp symbol NAME."
  (gethash name common-lisp-hyperspec--symbols))

(defun common-lisp-hyperspec--insert (name relative-url)
  "Insert CL symbol NAME and RELATIVE-URL into master table."
  (cl-pushnew relative-url
	      (gethash name common-lisp-hyperspec--symbols)
	      :test #'equal))

(defun common-lisp-hyperspec--strip-cl-package (name)
  (if (string-match "^\\([^:]*\\)::?\\([^:]*\\)$" name)
      (let ((package-name (match-string 1 name))
	    (symbol-name (match-string 2 name)))
	(if (member (downcase package-name)
		    '("cl" "common-lisp"))
	    symbol-name
	  name))
    name))

;; Choose the symbol at point or read symbol-name from the minibuffer.
(defun common-lisp-hyperspec-read-symbol-name (&optional symbol-at-point)
  (let* ((symbol-at-point (or symbol-at-point (thing-at-point 'symbol)))
	 (stripped-symbol (and symbol-at-point
			       (common-lisp-hyperspec--strip-cl-package
				(downcase symbol-at-point)))))
    (cond ((and stripped-symbol
		(common-lisp-hyperspec--find stripped-symbol))
	   stripped-symbol)
	  (t
	   (completing-read "Look up symbol in Common Lisp HyperSpec: "
			    common-lisp-hyperspec--symbols nil t
			    stripped-symbol
			    'common-lisp-hyperspec-history)))))

;; FIXME: is the (sleep-for 1.5) a actually needed?
(defun common-lisp-hyperspec (symbol-name)
  "View the documentation on SYMBOL-NAME from the Common Lisp HyperSpec.
If SYMBOL-NAME has more than one definition, all of them are displayed with
your favorite browser in sequence.  The browser should have a \"back\"
function to view the separate definitions.

The Common Lisp HyperSpec is the full ANSI Standard Common Lisp, provided
by Kent Pitman and Xanalys Inc.  By default, the Xanalys Web site is
visited to retrieve the information.  Xanalys Inc. allows you to transfer
the entire Common Lisp HyperSpec to your own site under certain conditions.
Visit http://www.lispworks.com/reference/HyperSpec/ for more information.
If you copy the HyperSpec to another location, customize the variable
`common-lisp-hyperspec-root' to point to that location."
  (interactive (list (common-lisp-hyperspec-read-symbol-name)))
  (let* ((name (common-lisp-hyperspec--strip-cl-package (downcase symbol-name)))
	 (hyperspec-root-path (if (string-suffix-p "/" common-lisp-hyperspec-root)
                                  common-lisp-hyperspec-root
                                (concat common-lisp-hyperspec-root "/"))))
    (cl-maplist (lambda (entry)
		  (browse-url (concat hyperspec-root-path "Body/" (car entry)))
		  (when (cdr entry)
		    (sleep-for 1.5)))
		(or (common-lisp-hyperspec--find name)
		    (error "The symbol `%s' is not defined in Common Lisp"
			   symbol-name)))))

;;; Added dynamic lookup of symbol in CLHS symbol table
;;;
;;; 20011202 Edi Weitz

;;; Replaced symbol table for v 4.0 with the one for v 6.0
;;; (which is now online at Xanalys' site)
;;;
;;; 20020213 Edi Weitz

(defun common-lisp-hyperspec--get-one-line ()
  (prog1
      (cl-delete ?\n (thing-at-point 'line))
    (forward-line)))

(defun common-lisp-hyperspec--parse-map-file (file)
  (with-temp-buffer
    (insert-file-contents file)
    (goto-char (point-min))
    (let ((result '()))
      (while (< (point) (point-max))
	(let* ((symbol-name (downcase (common-lisp-hyperspec--get-one-line)))
	       (relative-url (common-lisp-hyperspec--get-one-line))
	       (file (file-name-nondirectory relative-url)))
	  (push (list symbol-name file)
		result)))
      (reverse result))))

(mapc (lambda (entry)
	(common-lisp-hyperspec--insert (car entry) (cadr entry)))
      (if common-lisp-hyperspec-symbol-table
	  (common-lisp-hyperspec--parse-map-file
	   common-lisp-hyperspec-symbol-table)
	'(("&allow-other-keys" "03_da.htm")
	  ("&aux" "03_da.htm")
	  ("&body" "03_dd.htm")
	  ("&environment" "03_dd.htm")
	  ("&key" "03_da.htm")
	  ("&optional" "03_da.htm")
	  ("&rest" "03_da.htm")
	  ("&whole" "03_dd.htm")
	  ("*" "a_st.htm")
	  ("**" "v__stst_.htm")
	  ("***" "v__stst_.htm")
	  ("*break-on-signals*" "v_break_.htm")
	  ("*compile-file-pathname*" "v_cmp_fi.htm")
	  ("*compile-file-truename*" "v_cmp_fi.htm")
	  ("*compile-print*" "v_cmp_pr.htm")
	  ("*compile-verbose*" "v_cmp_pr.htm")
	  ("*debug-io*" "v_debug_.htm")
	  ("*debugger-hook*" "v_debugg.htm")
	  ("*default-pathname-defaults*" "v_defaul.htm")
	  ("*error-output*" "v_debug_.htm")
	  ("*features*" "v_featur.htm")
	  ("*gensym-counter*" "v_gensym.htm")
	  ("*load-pathname*" "v_ld_pns.htm")
	  ("*load-print*" "v_ld_prs.htm")
	  ("*load-truename*" "v_ld_pns.htm")
	  ("*load-verbose*" "v_ld_prs.htm")
	  ("*macroexpand-hook*" "v_mexp_h.htm")
	  ("*modules*" "v_module.htm")
	  ("*package*" "v_pkg.htm")
	  ("*print-array*" "v_pr_ar.htm")
	  ("*print-base*" "v_pr_bas.htm")
	  ("*print-case*" "v_pr_cas.htm")
	  ("*print-circle*" "v_pr_cir.htm")
	  ("*print-escape*" "v_pr_esc.htm")
	  ("*print-gensym*" "v_pr_gen.htm")
	  ("*print-length*" "v_pr_lev.htm")
	  ("*print-level*" "v_pr_lev.htm")
	  ("*print-lines*" "v_pr_lin.htm")
	  ("*print-miser-width*" "v_pr_mis.htm")
	  ("*print-pprint-dispatch*" "v_pr_ppr.htm")
	  ("*print-pretty*" "v_pr_pre.htm")
	  ("*print-radix*" "v_pr_bas.htm")
	  ("*print-readably*" "v_pr_rda.htm")
	  ("*print-right-margin*" "v_pr_rig.htm")
	  ("*query-io*" "v_debug_.htm")
	  ("*random-state*" "v_rnd_st.htm")
	  ("*read-base*" "v_rd_bas.htm")
	  ("*read-default-float-format*" "v_rd_def.htm")
	  ("*read-eval*" "v_rd_eva.htm")
	  ("*read-suppress*" "v_rd_sup.htm")
	  ("*readtable*" "v_rdtabl.htm")
	  ("*standard-input*" "v_debug_.htm")
	  ("*standard-output*" "v_debug_.htm")
	  ("*terminal-io*" "v_termin.htm")
	  ("*trace-output*" "v_debug_.htm")
	  ("+" "a_pl.htm")
	  ("++" "v_pl_plp.htm")
	  ("+++" "v_pl_plp.htm")
	  ("-" "a__.htm")
	  ("/" "a_sl.htm")
	  ("//" "v_sl_sls.htm")
	  ("///" "v_sl_sls.htm")
	  ("/=" "f_eq_sle.htm")
	  ("1+" "f_1pl_1_.htm")
	  ("1-" "f_1pl_1_.htm")
	  ("<" "f_eq_sle.htm")
	  ("<=" "f_eq_sle.htm")
	  ("=" "f_eq_sle.htm")
	  (">" "f_eq_sle.htm")
	  (">=" "f_eq_sle.htm")
	  ("abort" "a_abort.htm")
	  ("abs" "f_abs.htm")
	  ("acons" "f_acons.htm")
	  ("acos" "f_asin_.htm")
	  ("acosh" "f_sinh_.htm")
	  ("add-method" "f_add_me.htm")
	  ("adjoin" "f_adjoin.htm")
	  ("adjust-array" "f_adjust.htm")
	  ("adjustable-array-p" "f_adju_1.htm")
	  ("allocate-instance" "f_alloca.htm")
	  ("alpha-char-p" "f_alpha_.htm")
	  ("alphanumericp" "f_alphan.htm")
	  ("and" "a_and.htm")
	  ("append" "f_append.htm")
	  ("apply" "f_apply.htm")
	  ("apropos" "f_apropo.htm")
	  ("apropos-list" "f_apropo.htm")
	  ("aref" "f_aref.htm")
	  ("arithmetic-error" "e_arithm.htm")
	  ("arithmetic-error-operands" "f_arithm.htm")
	  ("arithmetic-error-operation" "f_arithm.htm")
	  ("array" "t_array.htm")
	  ("array-dimension" "f_ar_dim.htm")
	  ("array-dimension-limit" "v_ar_dim.htm")
	  ("array-dimensions" "f_ar_d_1.htm")
	  ("array-displacement" "f_ar_dis.htm")
	  ("array-element-type" "f_ar_ele.htm")
	  ("array-has-fill-pointer-p" "f_ar_has.htm")
	  ("array-in-bounds-p" "f_ar_in_.htm")
	  ("array-rank" "f_ar_ran.htm")
	  ("array-rank-limit" "v_ar_ran.htm")
	  ("array-row-major-index" "f_ar_row.htm")
	  ("array-total-size" "f_ar_tot.htm")
	  ("array-total-size-limit" "v_ar_tot.htm")
	  ("arrayp" "f_arrayp.htm")
	  ("ash" "f_ash.htm")
	  ("asin" "f_asin_.htm")
	  ("asinh" "f_sinh_.htm")
	  ("assert" "m_assert.htm")
	  ("assoc" "f_assocc.htm")
	  ("assoc-if" "f_assocc.htm")
	  ("assoc-if-not" "f_assocc.htm")
	  ("atan" "f_asin_.htm")
	  ("atanh" "f_sinh_.htm")
	  ("atom" "a_atom.htm")
	  ("base-char" "t_base_c.htm")
	  ("base-string" "t_base_s.htm")
	  ("bignum" "t_bignum.htm")
	  ("bit" "a_bit.htm")
	  ("bit-and" "f_bt_and.htm")
	  ("bit-andc1" "f_bt_and.htm")
	  ("bit-andc2" "f_bt_and.htm")
	  ("bit-eqv" "f_bt_and.htm")
	  ("bit-ior" "f_bt_and.htm")
	  ("bit-nand" "f_bt_and.htm")
	  ("bit-nor" "f_bt_and.htm")
	  ("bit-not" "f_bt_and.htm")
	  ("bit-orc1" "f_bt_and.htm")
	  ("bit-orc2" "f_bt_and.htm")
	  ("bit-vector" "t_bt_vec.htm")
	  ("bit-vector-p" "f_bt_vec.htm")
	  ("bit-xor" "f_bt_and.htm")
	  ("block" "s_block.htm")
	  ("boole" "f_boole.htm")
	  ("boole-1" "v_b_1_b.htm")
	  ("boole-2" "v_b_1_b.htm")
	  ("boole-and" "v_b_1_b.htm")
	  ("boole-andc1" "v_b_1_b.htm")
	  ("boole-andc2" "v_b_1_b.htm")
	  ("boole-c1" "v_b_1_b.htm")
	  ("boole-c2" "v_b_1_b.htm")
	  ("boole-clr" "v_b_1_b.htm")
	  ("boole-eqv" "v_b_1_b.htm")
	  ("boole-ior" "v_b_1_b.htm")
	  ("boole-nand" "v_b_1_b.htm")
	  ("boole-nor" "v_b_1_b.htm")
	  ("boole-orc1" "v_b_1_b.htm")
	  ("boole-orc2" "v_b_1_b.htm")
	  ("boole-set" "v_b_1_b.htm")
	  ("boole-xor" "v_b_1_b.htm")
	  ("boolean" "t_ban.htm")
	  ("both-case-p" "f_upper_.htm")
	  ("boundp" "f_boundp.htm")
	  ("break" "f_break.htm")
	  ("broadcast-stream" "t_broadc.htm")
	  ("broadcast-stream-streams" "f_broadc.htm")
	  ("built-in-class" "t_built_.htm")
	  ("butlast" "f_butlas.htm")
	  ("byte" "f_by_by.htm")
	  ("byte-position" "f_by_by.htm")
	  ("byte-size" "f_by_by.htm")
	  ("caaaar" "f_car_c.htm")
	  ("caaadr" "f_car_c.htm")
	  ("caaar" "f_car_c.htm")
	  ("caadar" "f_car_c.htm")
	  ("caaddr" "f_car_c.htm")
	  ("caadr" "f_car_c.htm")
	  ("caar" "f_car_c.htm")
	  ("cadaar" "f_car_c.htm")
	  ("cadadr" "f_car_c.htm")
	  ("cadar" "f_car_c.htm")
	  ("caddar" "f_car_c.htm")
	  ("cadddr" "f_car_c.htm")
	  ("caddr" "f_car_c.htm")
	  ("cadr" "f_car_c.htm")
	  ("call-arguments-limit" "v_call_a.htm")
	  ("call-method" "m_call_m.htm")
	  ("call-next-method" "f_call_n.htm")
	  ("car" "f_car_c.htm")
	  ("case" "m_case_.htm")
	  ("catch" "s_catch.htm")
	  ("ccase" "m_case_.htm")
	  ("cdaaar" "f_car_c.htm")
	  ("cdaadr" "f_car_c.htm")
	  ("cdaar" "f_car_c.htm")
	  ("cdadar" "f_car_c.htm")
	  ("cdaddr" "f_car_c.htm")
	  ("cdadr" "f_car_c.htm")
	  ("cdar" "f_car_c.htm")
	  ("cddaar" "f_car_c.htm")
	  ("cddadr" "f_car_c.htm")
	  ("cddar" "f_car_c.htm")
	  ("cdddar" "f_car_c.htm")
	  ("cddddr" "f_car_c.htm")
	  ("cdddr" "f_car_c.htm")
	  ("cddr" "f_car_c.htm")
	  ("cdr" "f_car_c.htm")
	  ("ceiling" "f_floorc.htm")
	  ("cell-error" "e_cell_e.htm")
	  ("cell-error-name" "f_cell_e.htm")
	  ("cerror" "f_cerror.htm")
	  ("change-class" "f_chg_cl.htm")
	  ("char" "f_char_.htm")
	  ("char-code" "f_char_c.htm")
	  ("char-code-limit" "v_char_c.htm")
	  ("char-downcase" "f_char_u.htm")
	  ("char-equal" "f_chareq.htm")
	  ("char-greaterp" "f_chareq.htm")
	  ("char-int" "f_char_i.htm")
	  ("char-lessp" "f_chareq.htm")
	  ("char-name" "f_char_n.htm")
	  ("char-not-equal" "f_chareq.htm")
	  ("char-not-greaterp" "f_chareq.htm")
	  ("char-not-lessp" "f_chareq.htm")
	  ("char-upcase" "f_char_u.htm")
	  ("char/=" "f_chareq.htm")
	  ("char<" "f_chareq.htm")
	  ("char<=" "f_chareq.htm")
	  ("char=" "f_chareq.htm")
	  ("char>" "f_chareq.htm")
	  ("char>=" "f_chareq.htm")
	  ("character" "a_ch.htm")
	  ("characterp" "f_chp.htm")
	  ("check-type" "m_check_.htm")
	  ("cis" "f_cis.htm")
	  ("class" "t_class.htm")
	  ("class-name" "f_class_.htm")
	  ("class-of" "f_clas_1.htm")
	  ("clear-input" "f_clear_.htm")
	  ("clear-output" "f_finish.htm")
	  ("close" "f_close.htm")
	  ("clrhash" "f_clrhas.htm")
	  ("code-char" "f_code_c.htm")
	  ("coerce" "f_coerce.htm")
	  ("compilation-speed" "d_optimi.htm")
	  ("compile" "f_cmp.htm")
	  ("compile-file" "f_cmp_fi.htm")
	  ("compile-file-pathname" "f_cmp__1.htm")
	  ("compiled-function" "t_cmpd_f.htm")
	  ("compiled-function-p" "f_cmpd_f.htm")
	  ("compiler-macro" "f_docume.htm")
	  ("compiler-macro-function" "f_cmp_ma.htm")
	  ("complement" "f_comple.htm")
	  ("complex" "a_comple.htm")
	  ("complexp" "f_comp_3.htm")
	  ("compute-applicable-methods" "f_comput.htm")
	  ("compute-restarts" "f_comp_1.htm")
	  ("concatenate" "f_concat.htm")
	  ("concatenated-stream" "t_concat.htm")
	  ("concatenated-stream-streams" "f_conc_1.htm")
	  ("cond" "m_cond.htm")
	  ("condition" "e_cnd.htm")
	  ("conjugate" "f_conjug.htm")
	  ("cons" "a_cons.htm")
	  ("consp" "f_consp.htm")
	  ("constantly" "f_cons_1.htm")
	  ("constantp" "f_consta.htm")
	  ("continue" "a_contin.htm")
	  ("control-error" "e_contro.htm")
	  ("copy-alist" "f_cp_ali.htm")
	  ("copy-list" "f_cp_lis.htm")
	  ("copy-pprint-dispatch" "f_cp_ppr.htm")
	  ("copy-readtable" "f_cp_rdt.htm")
	  ("copy-seq" "f_cp_seq.htm")
	  ("copy-structure" "f_cp_stu.htm")
	  ("copy-symbol" "f_cp_sym.htm")
	  ("copy-tree" "f_cp_tre.htm")
	  ("cos" "f_sin_c.htm")
	  ("cosh" "f_sinh_.htm")
	  ("count" "f_countc.htm")
	  ("count-if" "f_countc.htm")
	  ("count-if-not" "f_countc.htm")
	  ("ctypecase" "m_tpcase.htm")
	  ("debug" "d_optimi.htm")
	  ("decf" "m_incf_.htm")
	  ("declaim" "m_declai.htm")
	  ("declaration" "d_declar.htm")
	  ("declare" "s_declar.htm")
	  ("decode-float" "f_dec_fl.htm")
	  ("decode-universal-time" "f_dec_un.htm")
	  ("defclass" "m_defcla.htm")
	  ("defconstant" "m_defcon.htm")
	  ("defgeneric" "m_defgen.htm")
	  ("define-compiler-macro" "m_define.htm")
	  ("define-condition" "m_defi_5.htm")
	  ("define-method-combination" "m_defi_4.htm")
	  ("define-modify-macro" "m_defi_2.htm")
	  ("define-setf-expander" "m_defi_3.htm")
	  ("define-symbol-macro" "m_defi_1.htm")
	  ("defmacro" "m_defmac.htm")
	  ("defmethod" "m_defmet.htm")
	  ("defpackage" "m_defpkg.htm")
	  ("defparameter" "m_defpar.htm")
	  ("defsetf" "m_defset.htm")
	  ("defstruct" "m_defstr.htm")
	  ("deftype" "m_deftp.htm")
	  ("defun" "m_defun.htm")
	  ("defvar" "m_defpar.htm")
	  ("delete" "f_rm_rm.htm")
	  ("delete-duplicates" "f_rm_dup.htm")
	  ("delete-file" "f_del_fi.htm")
	  ("delete-if" "f_rm_rm.htm")
	  ("delete-if-not" "f_rm_rm.htm")
	  ("delete-package" "f_del_pk.htm")
	  ("denominator" "f_numera.htm")
	  ("deposit-field" "f_deposi.htm")
	  ("describe" "f_descri.htm")
	  ("describe-object" "f_desc_1.htm")
	  ("destructuring-bind" "m_destru.htm")
	  ("digit-char" "f_digit_.htm")
	  ("digit-char-p" "f_digi_1.htm")
	  ("directory" "f_dir.htm")
	  ("directory-namestring" "f_namest.htm")
	  ("disassemble" "f_disass.htm")
	  ("division-by-zero" "e_divisi.htm")
	  ("do" "m_do_do.htm")
	  ("do*" "m_do_do.htm")
	  ("do-all-symbols" "m_do_sym.htm")
	  ("do-external-symbols" "m_do_sym.htm")
	  ("do-symbols" "m_do_sym.htm")
	  ("documentation" "f_docume.htm")
	  ("dolist" "m_dolist.htm")
	  ("dotimes" "m_dotime.htm")
	  ("double-float" "t_short_.htm")
	  ("double-float-epsilon" "v_short_.htm")
	  ("double-float-negative-epsilon" "v_short_.htm")
	  ("dpb" "f_dpb.htm")
	  ("dribble" "f_dribbl.htm")
	  ("dynamic-extent" "d_dynami.htm")
	  ("ecase" "m_case_.htm")
	  ("echo-stream" "t_echo_s.htm")
	  ("echo-stream-input-stream" "f_echo_s.htm")
	  ("echo-stream-output-stream" "f_echo_s.htm")
	  ("ed" "f_ed.htm")
	  ("eighth" "f_firstc.htm")
	  ("elt" "f_elt.htm")
	  ("encode-universal-time" "f_encode.htm")
	  ("end-of-file" "e_end_of.htm")
	  ("endp" "f_endp.htm")
	  ("enough-namestring" "f_namest.htm")
	  ("ensure-directories-exist" "f_ensu_1.htm")
	  ("ensure-generic-function" "f_ensure.htm")
	  ("eq" "f_eq.htm")
	  ("eql" "a_eql.htm")
	  ("equal" "f_equal.htm")
	  ("equalp" "f_equalp.htm")
	  ("error" "a_error.htm")
	  ("etypecase" "m_tpcase.htm")
	  ("eval" "f_eval.htm")
	  ("eval-when" "s_eval_w.htm")
	  ("evenp" "f_evenpc.htm")
	  ("every" "f_everyc.htm")
	  ("exp" "f_exp_e.htm")
	  ("export" "f_export.htm")
	  ("expt" "f_exp_e.htm")
	  ("extended-char" "t_extend.htm")
	  ("fboundp" "f_fbound.htm")
	  ("fceiling" "f_floorc.htm")
	  ("fdefinition" "f_fdefin.htm")
	  ("ffloor" "f_floorc.htm")
	  ("fifth" "f_firstc.htm")
	  ("file-author" "f_file_a.htm")
	  ("file-error" "e_file_e.htm")
	  ("file-error-pathname" "f_file_e.htm")
	  ("file-length" "f_file_l.htm")
	  ("file-namestring" "f_namest.htm")
	  ("file-position" "f_file_p.htm")
	  ("file-stream" "t_file_s.htm")
	  ("file-string-length" "f_file_s.htm")
	  ("file-write-date" "f_file_w.htm")
	  ("fill" "f_fill.htm")
	  ("fill-pointer" "f_fill_p.htm")
	  ("find" "f_find_.htm")
	  ("find-all-symbols" "f_find_a.htm")
	  ("find-class" "f_find_c.htm")
	  ("find-if" "f_find_.htm")
	  ("find-if-not" "f_find_.htm")
	  ("find-method" "f_find_m.htm")
	  ("find-package" "f_find_p.htm")
	  ("find-restart" "f_find_r.htm")
	  ("find-symbol" "f_find_s.htm")
	  ("finish-output" "f_finish.htm")
	  ("first" "f_firstc.htm")
	  ("fixnum" "t_fixnum.htm")
	  ("flet" "s_flet_.htm")
	  ("float" "a_float.htm")
	  ("float-digits" "f_dec_fl.htm")
	  ("float-precision" "f_dec_fl.htm")
	  ("float-radix" "f_dec_fl.htm")
	  ("float-sign" "f_dec_fl.htm")
	  ("floating-point-inexact" "e_floa_1.htm")
	  ("floating-point-invalid-operation" "e_floati.htm")
	  ("floating-point-overflow" "e_floa_2.htm")
	  ("floating-point-underflow" "e_floa_3.htm")
	  ("floatp" "f_floatp.htm")
	  ("floor" "f_floorc.htm")
	  ("fmakunbound" "f_fmakun.htm")
	  ("force-output" "f_finish.htm")
	  ("format" "f_format.htm")
	  ("formatter" "m_format.htm")
	  ("fourth" "f_firstc.htm")
	  ("fresh-line" "f_terpri.htm")
	  ("fround" "f_floorc.htm")
	  ("ftruncate" "f_floorc.htm")
	  ("ftype" "d_ftype.htm")
	  ("funcall" "f_funcal.htm")
	  ("function" "a_fn.htm")
	  ("function-keywords" "f_fn_kwd.htm")
	  ("function-lambda-expression" "f_fn_lam.htm")
	  ("functionp" "f_fnp.htm")
	  ("gcd" "f_gcd.htm")
	  ("generic-function" "t_generi.htm")
	  ("gensym" "f_gensym.htm")
	  ("gentemp" "f_gentem.htm")
	  ("get" "f_get.htm")
	  ("get-decoded-time" "f_get_un.htm")
	  ("get-dispatch-macro-character" "f_set__1.htm")
	  ("get-internal-real-time" "f_get_in.htm")
	  ("get-internal-run-time" "f_get__1.htm")
	  ("get-macro-character" "f_set_ma.htm")
	  ("get-output-stream-string" "f_get_ou.htm")
	  ("get-properties" "f_get_pr.htm")
	  ("get-setf-expansion" "f_get_se.htm")
	  ("get-universal-time" "f_get_un.htm")
	  ("getf" "f_getf.htm")
	  ("gethash" "f_gethas.htm")
	  ("go" "s_go.htm")
	  ("graphic-char-p" "f_graphi.htm")
	  ("handler-bind" "m_handle.htm")
	  ("handler-case" "m_hand_1.htm")
	  ("hash-table" "t_hash_t.htm")
	  ("hash-table-count" "f_hash_1.htm")
	  ("hash-table-p" "f_hash_t.htm")
	  ("hash-table-rehash-size" "f_hash_2.htm")
	  ("hash-table-rehash-threshold" "f_hash_3.htm")
	  ("hash-table-size" "f_hash_4.htm")
	  ("hash-table-test" "f_hash_5.htm")
	  ("host-namestring" "f_namest.htm")
	  ("identity" "f_identi.htm")
	  ("if" "s_if.htm")
	  ("ignorable" "d_ignore.htm")
	  ("ignore" "d_ignore.htm")
	  ("ignore-errors" "m_ignore.htm")
	  ("imagpart" "f_realpa.htm")
	  ("import" "f_import.htm")
	  ("in-package" "m_in_pkg.htm")
	  ("incf" "m_incf_.htm")
	  ("initialize-instance" "f_init_i.htm")
	  ("inline" "d_inline.htm")
	  ("input-stream-p" "f_in_stm.htm")
	  ("inspect" "f_inspec.htm")
	  ("integer" "t_intege.htm")
	  ("integer-decode-float" "f_dec_fl.htm")
	  ("integer-length" "f_intege.htm")
	  ("integerp" "f_inte_1.htm")
	  ("interactive-stream-p" "f_intera.htm")
	  ("intern" "f_intern.htm")
	  ("internal-time-units-per-second" "v_intern.htm")
	  ("intersection" "f_isec_.htm")
	  ("invalid-method-error" "f_invali.htm")
	  ("invoke-debugger" "f_invoke.htm")
	  ("invoke-restart" "f_invo_1.htm")
	  ("invoke-restart-interactively" "f_invo_2.htm")
	  ("isqrt" "f_sqrt_.htm")
	  ("keyword" "t_kwd.htm")
	  ("keywordp" "f_kwdp.htm")
	  ("labels" "s_flet_.htm")
	  ("lambda" "a_lambda.htm")
	  ("lambda-list-keywords" "v_lambda.htm")
	  ("lambda-parameters-limit" "v_lamb_1.htm")
	  ("last" "f_last.htm")
	  ("lcm" "f_lcm.htm")
	  ("ldb" "f_ldb.htm")
	  ("ldb-test" "f_ldb_te.htm")
	  ("ldiff" "f_ldiffc.htm")
	  ("least-negative-double-float" "v_most_1.htm")
	  ("least-negative-long-float" "v_most_1.htm")
	  ("least-negative-normalized-double-float" "v_most_1.htm")
	  ("least-negative-normalized-long-float" "v_most_1.htm")
	  ("least-negative-normalized-short-float" "v_most_1.htm")
	  ("least-negative-normalized-single-float" "v_most_1.htm")
	  ("least-negative-short-float" "v_most_1.htm")
	  ("least-negative-single-float" "v_most_1.htm")
	  ("least-positive-double-float" "v_most_1.htm")
	  ("least-positive-long-float" "v_most_1.htm")
	  ("least-positive-normalized-double-float" "v_most_1.htm")
	  ("least-positive-normalized-long-float" "v_most_1.htm")
	  ("least-positive-normalized-short-float" "v_most_1.htm")
	  ("least-positive-normalized-single-float" "v_most_1.htm")
	  ("least-positive-short-float" "v_most_1.htm")
	  ("least-positive-single-float" "v_most_1.htm")
	  ("length" "f_length.htm")
	  ("let" "s_let_l.htm")
	  ("let*" "s_let_l.htm")
	  ("lisp-implementation-type" "f_lisp_i.htm")
	  ("lisp-implementation-version" "f_lisp_i.htm")
	  ("list" "a_list.htm")
	  ("list*" "f_list_.htm")
	  ("list-all-packages" "f_list_a.htm")
	  ("list-length" "f_list_l.htm")
	  ("listen" "f_listen.htm")
	  ("listp" "f_listp.htm")
	  ("load" "f_load.htm")
	  ("load-logical-pathname-translations" "f_ld_log.htm")
	  ("load-time-value" "s_ld_tim.htm")
	  ("locally" "s_locall.htm")
	  ("log" "f_log.htm")
	  ("logand" "f_logand.htm")
	  ("logandc1" "f_logand.htm")
	  ("logandc2" "f_logand.htm")
	  ("logbitp" "f_logbtp.htm")
	  ("logcount" "f_logcou.htm")
	  ("logeqv" "f_logand.htm")
	  ("logical-pathname" "a_logica.htm")
	  ("logical-pathname-translations" "f_logica.htm")
	  ("logior" "f_logand.htm")
	  ("lognand" "f_logand.htm")
	  ("lognor" "f_logand.htm")
	  ("lognot" "f_logand.htm")
	  ("logorc1" "f_logand.htm")
	  ("logorc2" "f_logand.htm")
	  ("logtest" "f_logtes.htm")
	  ("logxor" "f_logand.htm")
	  ("long-float" "t_short_.htm")
	  ("long-float-epsilon" "v_short_.htm")
	  ("long-float-negative-epsilon" "v_short_.htm")
	  ("long-site-name" "f_short_.htm")
	  ("loop" "m_loop.htm")
	  ("loop-finish" "m_loop_f.htm")
	  ("lower-case-p" "f_upper_.htm")
	  ("machine-instance" "f_mach_i.htm")
	  ("machine-type" "f_mach_t.htm")
	  ("machine-version" "f_mach_v.htm")
	  ("macro-function" "f_macro_.htm")
	  ("macroexpand" "f_mexp_.htm")
	  ("macroexpand-1" "f_mexp_.htm")
	  ("macrolet" "s_flet_.htm")
	  ("make-array" "f_mk_ar.htm")
	  ("make-broadcast-stream" "f_mk_bro.htm")
	  ("make-concatenated-stream" "f_mk_con.htm")
	  ("make-condition" "f_mk_cnd.htm")
	  ("make-dispatch-macro-character" "f_mk_dis.htm")
	  ("make-echo-stream" "f_mk_ech.htm")
	  ("make-hash-table" "f_mk_has.htm")
	  ("make-instance" "f_mk_ins.htm")
	  ("make-instances-obsolete" "f_mk_i_1.htm")
	  ("make-list" "f_mk_lis.htm")
	  ("make-load-form" "f_mk_ld_.htm")
	  ("make-load-form-saving-slots" "f_mk_l_1.htm")
	  ("make-method" "m_call_m.htm")
	  ("make-package" "f_mk_pkg.htm")
	  ("make-pathname" "f_mk_pn.htm")
	  ("make-random-state" "f_mk_rnd.htm")
	  ("make-sequence" "f_mk_seq.htm")
	  ("make-string" "f_mk_stg.htm")
	  ("make-string-input-stream" "f_mk_s_1.htm")
	  ("make-string-output-stream" "f_mk_s_2.htm")
	  ("make-symbol" "f_mk_sym.htm")
	  ("make-synonym-stream" "f_mk_syn.htm")
	  ("make-two-way-stream" "f_mk_two.htm")
	  ("makunbound" "f_makunb.htm")
	  ("map" "f_map.htm")
	  ("map-into" "f_map_in.htm")
	  ("mapc" "f_mapc_.htm")
	  ("mapcan" "f_mapc_.htm")
	  ("mapcar" "f_mapc_.htm")
	  ("mapcon" "f_mapc_.htm")
	  ("maphash" "f_maphas.htm")
	  ("mapl" "f_mapc_.htm")
	  ("maplist" "f_mapc_.htm")
	  ("mask-field" "f_mask_f.htm")
	  ("max" "f_max_m.htm")
	  ("member" "a_member.htm")
	  ("member-if" "f_mem_m.htm")
	  ("member-if-not" "f_mem_m.htm")
	  ("merge" "f_merge.htm")
	  ("merge-pathnames" "f_merge_.htm")
	  ("method" "t_method.htm")
	  ("method-combination" "a_method.htm")
	  ("method-combination-error" "f_meth_1.htm")
	  ("method-qualifiers" "f_method.htm")
	  ("min" "f_max_m.htm")
	  ("minusp" "f_minusp.htm")
	  ("mismatch" "f_mismat.htm")
	  ("mod" "a_mod.htm")
	  ("most-negative-double-float" "v_most_1.htm")
	  ("most-negative-fixnum" "v_most_p.htm")
	  ("most-negative-long-float" "v_most_1.htm")
	  ("most-negative-short-float" "v_most_1.htm")
	  ("most-negative-single-float" "v_most_1.htm")
	  ("most-positive-double-float" "v_most_1.htm")
	  ("most-positive-fixnum" "v_most_p.htm")
	  ("most-positive-long-float" "v_most_1.htm")
	  ("most-positive-short-float" "v_most_1.htm")
	  ("most-positive-single-float" "v_most_1.htm")
	  ("muffle-warning" "a_muffle.htm")
	  ("multiple-value-bind" "m_multip.htm")
	  ("multiple-value-call" "s_multip.htm")
	  ("multiple-value-list" "m_mult_1.htm")
	  ("multiple-value-prog1" "s_mult_1.htm")
	  ("multiple-value-setq" "m_mult_2.htm")
	  ("multiple-values-limit" "v_multip.htm")
	  ("name-char" "f_name_c.htm")
	  ("namestring" "f_namest.htm")
	  ("nbutlast" "f_butlas.htm")
	  ("nconc" "f_nconc.htm")
	  ("next-method-p" "f_next_m.htm")
	  ("nil" "a_nil.htm")
	  ("nintersection" "f_isec_.htm")
	  ("ninth" "f_firstc.htm")
	  ("no-applicable-method" "f_no_app.htm")
	  ("no-next-method" "f_no_nex.htm")
	  ("not" "a_not.htm")
	  ("notany" "f_everyc.htm")
	  ("notevery" "f_everyc.htm")
	  ("notinline" "d_inline.htm")
	  ("nreconc" "f_revapp.htm")
	  ("nreverse" "f_revers.htm")
	  ("nset-difference" "f_set_di.htm")
	  ("nset-exclusive-or" "f_set_ex.htm")
	  ("nstring-capitalize" "f_stg_up.htm")
	  ("nstring-downcase" "f_stg_up.htm")
	  ("nstring-upcase" "f_stg_up.htm")
	  ("nsublis" "f_sublis.htm")
	  ("nsubst" "f_substc.htm")
	  ("nsubst-if" "f_substc.htm")
	  ("nsubst-if-not" "f_substc.htm")
	  ("nsubstitute" "f_sbs_s.htm")
	  ("nsubstitute-if" "f_sbs_s.htm")
	  ("nsubstitute-if-not" "f_sbs_s.htm")
	  ("nth" "f_nth.htm")
	  ("nth-value" "m_nth_va.htm")
	  ("nthcdr" "f_nthcdr.htm")
	  ("null" "a_null.htm")
	  ("number" "t_number.htm")
	  ("numberp" "f_nump.htm")
	  ("numerator" "f_numera.htm")
	  ("nunion" "f_unionc.htm")
	  ("oddp" "f_evenpc.htm")
	  ("open" "f_open.htm")
	  ("open-stream-p" "f_open_s.htm")
	  ("optimize" "d_optimi.htm")
	  ("or" "a_or.htm")
	  ("otherwise" "m_case_.htm")
	  ("output-stream-p" "f_in_stm.htm")
	  ("package" "t_pkg.htm")
	  ("package-error" "e_pkg_er.htm")
	  ("package-error-package" "f_pkg_er.htm")
	  ("package-name" "f_pkg_na.htm")
	  ("package-nicknames" "f_pkg_ni.htm")
	  ("package-shadowing-symbols" "f_pkg_sh.htm")
	  ("package-use-list" "f_pkg_us.htm")
	  ("package-used-by-list" "f_pkg__1.htm")
	  ("packagep" "f_pkgp.htm")
	  ("pairlis" "f_pairli.htm")
	  ("parse-error" "e_parse_.htm")
	  ("parse-integer" "f_parse_.htm")
	  ("parse-namestring" "f_pars_1.htm")
	  ("pathname" "a_pn.htm")
	  ("pathname-device" "f_pn_hos.htm")
	  ("pathname-directory" "f_pn_hos.htm")
	  ("pathname-host" "f_pn_hos.htm")
	  ("pathname-match-p" "f_pn_mat.htm")
	  ("pathname-name" "f_pn_hos.htm")
	  ("pathname-type" "f_pn_hos.htm")
	  ("pathname-version" "f_pn_hos.htm")
	  ("pathnamep" "f_pnp.htm")
	  ("peek-char" "f_peek_c.htm")
	  ("phase" "f_phase.htm")
	  ("pi" "v_pi.htm")
	  ("plusp" "f_minusp.htm")
	  ("pop" "m_pop.htm")
	  ("position" "f_pos_p.htm")
	  ("position-if" "f_pos_p.htm")
	  ("position-if-not" "f_pos_p.htm")
	  ("pprint" "f_wr_pr.htm")
	  ("pprint-dispatch" "f_ppr_di.htm")
	  ("pprint-exit-if-list-exhausted" "m_ppr_ex.htm")
	  ("pprint-fill" "f_ppr_fi.htm")
	  ("pprint-indent" "f_ppr_in.htm")
	  ("pprint-linear" "f_ppr_fi.htm")
	  ("pprint-logical-block" "m_ppr_lo.htm")
	  ("pprint-newline" "f_ppr_nl.htm")
	  ("pprint-pop" "m_ppr_po.htm")
	  ("pprint-tab" "f_ppr_ta.htm")
	  ("pprint-tabular" "f_ppr_fi.htm")
	  ("prin1" "f_wr_pr.htm")
	  ("prin1-to-string" "f_wr_to_.htm")
	  ("princ" "f_wr_pr.htm")
	  ("princ-to-string" "f_wr_to_.htm")
	  ("print" "f_wr_pr.htm")
	  ("print-not-readable" "e_pr_not.htm")
	  ("print-not-readable-object" "f_pr_not.htm")
	  ("print-object" "f_pr_obj.htm")
	  ("print-unreadable-object" "m_pr_unr.htm")
	  ("probe-file" "f_probe_.htm")
	  ("proclaim" "f_procla.htm")
	  ("prog" "m_prog_.htm")
	  ("prog*" "m_prog_.htm")
	  ("prog1" "m_prog1c.htm")
	  ("prog2" "m_prog1c.htm")
	  ("progn" "s_progn.htm")
	  ("program-error" "e_progra.htm")
	  ("progv" "s_progv.htm")
	  ("provide" "f_provid.htm")
	  ("psetf" "m_setf_.htm")
	  ("psetq" "m_psetq.htm")
	  ("push" "m_push.htm")
	  ("pushnew" "m_pshnew.htm")
	  ("quote" "s_quote.htm")
	  ("random" "f_random.htm")
	  ("random-state" "t_rnd_st.htm")
	  ("random-state-p" "f_rnd_st.htm")
	  ("rassoc" "f_rassoc.htm")
	  ("rassoc-if" "f_rassoc.htm")
	  ("rassoc-if-not" "f_rassoc.htm")
	  ("ratio" "t_ratio.htm")
	  ("rational" "a_ration.htm")
	  ("rationalize" "f_ration.htm")
	  ("rationalp" "f_rati_1.htm")
	  ("read" "f_rd_rd.htm")
	  ("read-byte" "f_rd_by.htm")
	  ("read-char" "f_rd_cha.htm")
	  ("read-char-no-hang" "f_rd_c_1.htm")
	  ("read-delimited-list" "f_rd_del.htm")
	  ("read-from-string" "f_rd_fro.htm")
	  ("read-line" "f_rd_lin.htm")
	  ("read-preserving-whitespace" "f_rd_rd.htm")
	  ("read-sequence" "f_rd_seq.htm")
	  ("reader-error" "e_rder_e.htm")
	  ("readtable" "t_rdtabl.htm")
	  ("readtable-case" "f_rdtabl.htm")
	  ("readtablep" "f_rdta_1.htm")
	  ("real" "t_real.htm")
	  ("realp" "f_realp.htm")
	  ("realpart" "f_realpa.htm")
	  ("reduce" "f_reduce.htm")
	  ("reinitialize-instance" "f_reinit.htm")
	  ("rem" "f_mod_r.htm")
	  ("remf" "m_remf.htm")
	  ("remhash" "f_remhas.htm")
	  ("remove" "f_rm_rm.htm")
	  ("remove-duplicates" "f_rm_dup.htm")
	  ("remove-if" "f_rm_rm.htm")
	  ("remove-if-not" "f_rm_rm.htm")
	  ("remove-method" "f_rm_met.htm")
	  ("remprop" "f_rempro.htm")
	  ("rename-file" "f_rn_fil.htm")
	  ("rename-package" "f_rn_pkg.htm")
	  ("replace" "f_replac.htm")
	  ("require" "f_provid.htm")
	  ("rest" "f_rest.htm")
	  ("restart" "t_rst.htm")
	  ("restart-bind" "m_rst_bi.htm")
	  ("restart-case" "m_rst_ca.htm")
	  ("restart-name" "f_rst_na.htm")
	  ("return" "m_return.htm")
	  ("return-from" "s_ret_fr.htm")
	  ("revappend" "f_revapp.htm")
	  ("reverse" "f_revers.htm")
	  ("room" "f_room.htm")
	  ("rotatef" "m_rotate.htm")
	  ("round" "f_floorc.htm")
	  ("row-major-aref" "f_row_ma.htm")
	  ("rplaca" "f_rplaca.htm")
	  ("rplacd" "f_rplaca.htm")
	  ("safety" "d_optimi.htm")
	  ("satisfies" "t_satisf.htm")
	  ("sbit" "f_bt_sb.htm")
	  ("scale-float" "f_dec_fl.htm")
	  ("schar" "f_char_.htm")
	  ("search" "f_search.htm")
	  ("second" "f_firstc.htm")
	  ("sequence" "t_seq.htm")
	  ("serious-condition" "e_seriou.htm")
	  ("set" "f_set.htm")
	  ("set-difference" "f_set_di.htm")
	  ("set-dispatch-macro-character" "f_set__1.htm")
	  ("set-exclusive-or" "f_set_ex.htm")
	  ("set-macro-character" "f_set_ma.htm")
	  ("set-pprint-dispatch" "f_set_pp.htm")
	  ("set-syntax-from-char" "f_set_sy.htm")
	  ("setf" "a_setf.htm")
	  ("setq" "s_setq.htm")
	  ("seventh" "f_firstc.htm")
	  ("shadow" "f_shadow.htm")
	  ("shadowing-import" "f_shdw_i.htm")
	  ("shared-initialize" "f_shared.htm")
	  ("shiftf" "m_shiftf.htm")
	  ("short-float" "t_short_.htm")
	  ("short-float-epsilon" "v_short_.htm")
	  ("short-float-negative-epsilon" "v_short_.htm")
	  ("short-site-name" "f_short_.htm")
	  ("signal" "f_signal.htm")
	  ("signed-byte" "t_sgn_by.htm")
	  ("signum" "f_signum.htm")
	  ("simple-array" "t_smp_ar.htm")
	  ("simple-base-string" "t_smp_ba.htm")
	  ("simple-bit-vector" "t_smp_bt.htm")
	  ("simple-bit-vector-p" "f_smp_bt.htm")
	  ("simple-condition" "e_smp_cn.htm")
	  ("simple-condition-format-arguments" "f_smp_cn.htm")
	  ("simple-condition-format-control" "f_smp_cn.htm")
	  ("simple-error" "e_smp_er.htm")
	  ("simple-string" "t_smp_st.htm")
	  ("simple-string-p" "f_smp_st.htm")
	  ("simple-type-error" "e_smp_tp.htm")
	  ("simple-vector" "t_smp_ve.htm")
	  ("simple-vector-p" "f_smp_ve.htm")
	  ("simple-warning" "e_smp_wa.htm")
	  ("sin" "f_sin_c.htm")
	  ("single-float" "t_short_.htm")
	  ("single-float-epsilon" "v_short_.htm")
	  ("single-float-negative-epsilon" "v_short_.htm")
	  ("sinh" "f_sinh_.htm")
	  ("sixth" "f_firstc.htm")
	  ("sleep" "f_sleep.htm")
	  ("slot-boundp" "f_slt_bo.htm")
	  ("slot-exists-p" "f_slt_ex.htm")
	  ("slot-makunbound" "f_slt_ma.htm")
	  ("slot-missing" "f_slt_mi.htm")
	  ("slot-unbound" "f_slt_un.htm")
	  ("slot-value" "f_slt_va.htm")
	  ("software-type" "f_sw_tpc.htm")
	  ("software-version" "f_sw_tpc.htm")
	  ("some" "f_everyc.htm")
	  ("sort" "f_sort_.htm")
	  ("space" "d_optimi.htm")
	  ("special" "d_specia.htm")
	  ("special-operator-p" "f_specia.htm")
	  ("speed" "d_optimi.htm")
	  ("sqrt" "f_sqrt_.htm")
	  ("stable-sort" "f_sort_.htm")
	  ("standard" "07_ffb.htm")
	  ("standard-char" "t_std_ch.htm")
	  ("standard-char-p" "f_std_ch.htm")
	  ("standard-class" "t_std_cl.htm")
	  ("standard-generic-function" "t_std_ge.htm")
	  ("standard-method" "t_std_me.htm")
	  ("standard-object" "t_std_ob.htm")
	  ("step" "m_step.htm")
	  ("storage-condition" "e_storag.htm")
	  ("store-value" "a_store_.htm")
	  ("stream" "t_stream.htm")
	  ("stream-element-type" "f_stm_el.htm")
	  ("stream-error" "e_stm_er.htm")
	  ("stream-error-stream" "f_stm_er.htm")
	  ("stream-external-format" "f_stm_ex.htm")
	  ("streamp" "f_stmp.htm")
	  ("string" "a_string.htm")
	  ("string-capitalize" "f_stg_up.htm")
	  ("string-downcase" "f_stg_up.htm")
	  ("string-equal" "f_stgeq_.htm")
	  ("string-greaterp" "f_stgeq_.htm")
	  ("string-left-trim" "f_stg_tr.htm")
	  ("string-lessp" "f_stgeq_.htm")
	  ("string-not-equal" "f_stgeq_.htm")
	  ("string-not-greaterp" "f_stgeq_.htm")
	  ("string-not-lessp" "f_stgeq_.htm")
	  ("string-right-trim" "f_stg_tr.htm")
	  ("string-stream" "t_stg_st.htm")
	  ("string-trim" "f_stg_tr.htm")
	  ("string-upcase" "f_stg_up.htm")
	  ("string/=" "f_stgeq_.htm")
	  ("string<" "f_stgeq_.htm")
	  ("string<=" "f_stgeq_.htm")
	  ("string=" "f_stgeq_.htm")
	  ("string>" "f_stgeq_.htm")
	  ("string>=" "f_stgeq_.htm")
	  ("stringp" "f_stgp.htm")
	  ("structure" "f_docume.htm")
	  ("structure-class" "t_stu_cl.htm")
	  ("structure-object" "t_stu_ob.htm")
	  ("style-warning" "e_style_.htm")
	  ("sublis" "f_sublis.htm")
	  ("subseq" "f_subseq.htm")
	  ("subsetp" "f_subset.htm")
	  ("subst" "f_substc.htm")
	  ("subst-if" "f_substc.htm")
	  ("subst-if-not" "f_substc.htm")
	  ("substitute" "f_sbs_s.htm")
	  ("substitute-if" "f_sbs_s.htm")
	  ("substitute-if-not" "f_sbs_s.htm")
	  ("subtypep" "f_subtpp.htm")
	  ("svref" "f_svref.htm")
	  ("sxhash" "f_sxhash.htm")
	  ("symbol" "t_symbol.htm")
	  ("symbol-function" "f_symb_1.htm")
	  ("symbol-macrolet" "s_symbol.htm")
	  ("symbol-name" "f_symb_2.htm")
	  ("symbol-package" "f_symb_3.htm")
	  ("symbol-plist" "f_symb_4.htm")
	  ("symbol-value" "f_symb_5.htm")
	  ("symbolp" "f_symbol.htm")
	  ("synonym-stream" "t_syn_st.htm")
	  ("synonym-stream-symbol" "f_syn_st.htm")
	  ("t" "a_t.htm")
	  ("tagbody" "s_tagbod.htm")
	  ("tailp" "f_ldiffc.htm")
	  ("tan" "f_sin_c.htm")
	  ("tanh" "f_sinh_.htm")
	  ("tenth" "f_firstc.htm")
	  ("terpri" "f_terpri.htm")
	  ("the" "s_the.htm")
	  ("third" "f_firstc.htm")
	  ("throw" "s_throw.htm")
	  ("time" "m_time.htm")
	  ("trace" "m_tracec.htm")
	  ("translate-logical-pathname" "f_tr_log.htm")
	  ("translate-pathname" "f_tr_pn.htm")
	  ("tree-equal" "f_tree_e.htm")
	  ("truename" "f_tn.htm")
	  ("truncate" "f_floorc.htm")
	  ("two-way-stream" "t_two_wa.htm")
	  ("two-way-stream-input-stream" "f_two_wa.htm")
	  ("two-way-stream-output-stream" "f_two_wa.htm")
	  ("type" "a_type.htm")
	  ("type-error" "e_tp_err.htm")
	  ("type-error-datum" "f_tp_err.htm")
	  ("type-error-expected-type" "f_tp_err.htm")
	  ("type-of" "f_tp_of.htm")
	  ("typecase" "m_tpcase.htm")
	  ("typep" "f_typep.htm")
	  ("unbound-slot" "e_unboun.htm")
	  ("unbound-slot-instance" "f_unboun.htm")
	  ("unbound-variable" "e_unbo_1.htm")
	  ("undefined-function" "e_undefi.htm")
	  ("unexport" "f_unexpo.htm")
	  ("unintern" "f_uninte.htm")
	  ("union" "f_unionc.htm")
	  ("unless" "m_when_.htm")
	  ("unread-char" "f_unrd_c.htm")
	  ("unsigned-byte" "t_unsgn_.htm")
	  ("untrace" "m_tracec.htm")
	  ("unuse-package" "f_unuse_.htm")
	  ("unwind-protect" "s_unwind.htm")
	  ("update-instance-for-different-class" "f_update.htm")
	  ("update-instance-for-redefined-class" "f_upda_1.htm")
	  ("upgraded-array-element-type" "f_upgr_1.htm")
	  ("upgraded-complex-part-type" "f_upgrad.htm")
	  ("upper-case-p" "f_upper_.htm")
	  ("use-package" "f_use_pk.htm")
	  ("use-value" "a_use_va.htm")
	  ("user-homedir-pathname" "f_user_h.htm")
	  ("values" "a_values.htm")
	  ("values-list" "f_vals_l.htm")
	  ("variable" "f_docume.htm")
	  ("vector" "a_vector.htm")
	  ("vector-pop" "f_vec_po.htm")
	  ("vector-push" "f_vec_ps.htm")
	  ("vector-push-extend" "f_vec_ps.htm")
	  ("vectorp" "f_vecp.htm")
	  ("warn" "f_warn.htm")
	  ("warning" "e_warnin.htm")
	  ("when" "m_when_.htm")
	  ("wild-pathname-p" "f_wild_p.htm")
	  ("with-accessors" "m_w_acce.htm")
	  ("with-compilation-unit" "m_w_comp.htm")
	  ("with-condition-restarts" "m_w_cnd_.htm")
	  ("with-hash-table-iterator" "m_w_hash.htm")
	  ("with-input-from-string" "m_w_in_f.htm")
	  ("with-open-file" "m_w_open.htm")
	  ("with-open-stream" "m_w_op_1.htm")
	  ("with-output-to-string" "m_w_out_.htm")
	  ("with-package-iterator" "m_w_pkg_.htm")
	  ("with-simple-restart" "m_w_smp_.htm")
	  ("with-slots" "m_w_slts.htm")
	  ("with-standard-io-syntax" "m_w_std_.htm")
	  ("write" "f_wr_pr.htm")
	  ("write-byte" "f_wr_by.htm")
	  ("write-char" "f_wr_cha.htm")
	  ("write-line" "f_wr_stg.htm")
	  ("write-sequence" "f_wr_seq.htm")
	  ("write-string" "f_wr_stg.htm")
	  ("write-to-string" "f_wr_to_.htm")
	  ("y-or-n-p" "f_y_or_n.htm")
	  ("yes-or-no-p" "f_y_or_n.htm")
	  ("zerop" "f_zerop.htm"))))

;;; Added entries for reader macros.
;;;
;;; 20090302 Tobias C Rittweiler, and Stas Boukarev

(defvar common-lisp-hyperspec--reader-macros (make-hash-table :test #'equal))

;;; Data/Map_Sym.txt in does not contain entries for the reader
;;; macros. So we have to enumerate these explicitly.
(mapc (lambda (entry)
	(puthash (car entry) (cadr entry)
		 common-lisp-hyperspec--reader-macros))
      '(("#" "02_dh.htm")
	("##" "02_dhp.htm")
	("#'" "02_dhb.htm")
	("#(" "02_dhc.htm")
	("#*" "02_dhd.htm")
	("#:" "02_dhe.htm")
	("#." "02_dhf.htm")
	("#=" "02_dho.htm")
	("#+" "02_dhq.htm")
	("#-" "02_dhr.htm")
	("#<" "02_dht.htm")
	("#A" "02_dhl.htm")
	("#B" "02_dhg.htm")
	("#C" "02_dhk.htm")
	("#O" "02_dhh.htm")
	("#P" "02_dhn.htm")
	("#R" "02_dhj.htm")
	("#S" "02_dhm.htm")
	("#X" "02_dhi.htm")
	("#\\" "02_dha.htm")
	("#|" "02_dhs.htm")
	("\"" "02_de.htm")
	("'" "02_dc.htm")
	("`" "02_df.htm")
	("," "02_dg.htm")
	("(" "02_da.htm")
	(")" "02_db.htm")
	(";" "02_dd.htm")))

(defun common-lisp-hyperspec-lookup-reader-macro (macro)
  "Browse the CLHS entry for the reader-macro MACRO."
  (interactive
   (list
    (let ((completion-ignore-case t))
      (completing-read "Look up reader-macro: "
		       common-lisp-hyperspec--reader-macros nil t
		       (common-lisp-hyperspec-reader-macro-at-point)))))
  (browse-url
   (concat common-lisp-hyperspec-root "Body/"
	   (gethash macro common-lisp-hyperspec--reader-macros))))

(defun common-lisp-hyperspec-reader-macro-at-point ()
  (let ((regexp "\\(#.?\\)\\|\\([\"',`';()]\\)"))
    (when (looking-back regexp nil t)
      (match-string-no-properties 0))))

;;; FORMAT character lookup by Frode Vatvedt Fjeld <frodef@acm.org> 20030902
;;;
;;; adjusted for ILISP by Nikodemus Siivola 20030903

(defvar common-lisp-hyperspec-format-history nil
  "History of format characters looked up in the Common Lisp HyperSpec.")

(defun common-lisp-hyperspec-section-6.0 (indices)
  (let ((string (format "%sBody/%s_"
			common-lisp-hyperspec-root
			(let ((base (pop indices)))
			  (if (< base 10)
			      (format "0%s" base)
			    base)))))
    (concat string
	    (mapconcat (lambda (n)
			 (make-string 1 (+ ?a (- n 1))))
		       indices
		       "")
	    ".htm")))

(defun common-lisp-hyperspec-section-4.0 (indices)
  (let ((string (format "%sBody/sec_"
			common-lisp-hyperspec-root)))
    (concat string
	    (mapconcat (lambda (n)
			 (format "%d" n))
		       indices
		       "-")
	    ".html")))

(defvar common-lisp-hyperspec-section-fun 'common-lisp-hyperspec-section-6.0)

(defun common-lisp-hyperspec-section (indices)
  (funcall common-lisp-hyperspec-section-fun indices))

(defvar common-lisp-hyperspec--format-characters
  (make-hash-table :test 'equal))

(defun common-lisp-hyperspec--read-format-character ()
  (let ((char-at-point
	 (ignore-errors (char-to-string (char-after (point))))))
    (if (and char-at-point
	     (gethash (upcase char-at-point)
		      common-lisp-hyperspec--format-characters))
	char-at-point
      (completing-read
       "Look up format control character in Common Lisp HyperSpec: "
       common-lisp-hyperspec--format-characters nil t nil
       'common-lisp-hyperspec-format-history))))

(defun common-lisp-hyperspec-format (character-name)
  (interactive (list (common-lisp-hyperspec--read-format-character)))
  (cl-maplist (lambda (entry)
		(browse-url (common-lisp-hyperspec-section (car entry))))
	      (or (gethash character-name
			   common-lisp-hyperspec--format-characters)
		  (error "The symbol `%s' is not defined in Common Lisp"
			 character-name))))

;;; Previously there were entries for "C" and "C: Character",
;;; which unpleasingly crowded the completion buffer, so I made
;;; it show one entry ("C - Character") only.
;;;
;;; 20100131 Tobias C Rittweiler

(defun common-lisp-hyperspec--insert-format-directive (char section
							    &optional summary)
  (let* ((designator (if summary (format "%s - %s" char summary) char)))
    (cl-pushnew section (gethash designator
				 common-lisp-hyperspec--format-characters)
		:test #'equal)))

(mapc (lambda (entry)
	(cl-destructuring-bind (char section &optional summary) entry
	  (common-lisp-hyperspec--insert-format-directive char section summary)
	  (when (and (= 1 (length char))
		     (not (string-equal char (upcase char))))
	    (common-lisp-hyperspec--insert-format-directive
	     (upcase char) section summary))))
      '(("c" (22 3 1 1) "Character")
	("%" (22 3 1 2) "Newline")
	("&" (22 3 1 3) "Fresh-line")
	("|" (22 3 1 4) "Page")
	("~" (22 3 1 5) "Tilde")
	("r" (22 3 2 1) "Radix")
	("d" (22 3 2 2) "Decimal")
	("b" (22 3 2 3) "Binary")
	("o" (22 3 2 4) "Octal")
	("x" (22 3 2 5) "Hexadecimal")
	("f" (22 3 3 1) "Fixed-Format Floating-Point")
	("e" (22 3 3 2) "Exponential Floating-Point")
	("g" (22 3 3 3) "General Floating-Point")
	("$" (22 3 3 4) "Monetary Floating-Point")
	("a" (22 3 4 1) "Aesthetic")
	("s" (22 3 4 2) "Standard")
	("w" (22 3 4 3) "Write")
	("_" (22 3 5 1) "Conditional Newline")
	("<" (22 3 5 2) "Logical Block")
	("i" (22 3 5 3) "Indent")
	("/" (22 3 5 4) "Call Function")
	("t" (22 3 6 1) "Tabulate")
	("<" (22 3 6 2) "Justification")
	(">" (22 3 6 3) "End of Justification")
	("*" (22 3 7 1) "Go-To")
	("[" (22 3 7 2) "Conditional Expression")
	("]" (22 3 7 3) "End of Conditional Expression")
	("{" (22 3 7 4) "Iteration")
	("}" (22 3 7 5) "End of Iteration")
	("?" (22 3 7 6) "Recursive Processing")
	("(" (22 3 8 1) "Case Conversion")
	(")" (22 3 8 2) "End of Case Conversion")
	("p" (22 3 8 3) "Plural")
	(";" (22 3 9 1) "Clause Separator")
	("^" (22 3 9 2) "Escape Upward")
	("Newline: Ignored Newline" (22 3 9 3))
	("Nesting of FORMAT Operations" (22 3 10 1))
	("Missing and Additional FORMAT Arguments" (22 3 10 2))
	("Additional FORMAT Parameters" (22 3 10 3))))


;;;; Glossary

(defvar common-lisp-hyperspec-glossary-function 'common-lisp-glossary-6.0
  "Function that creates a URL for a glossary term.")

(define-obsolete-variable-alias 'common-lisp-glossary-fun
  'common-lisp-hyperspec-glossary-function "2015-12-29")

(defvar common-lisp-hyperspec--glossary-terms (make-hash-table :test #'equal)
  "Collection of glossary terms and relative URLs.")

;;; Functions

;;; The functions below are used to collect glossary terms and page anchors
;;; from CLHS. They are commented out because they are not needed unless the
;;; list of terms/anchors need to be updated.

;; (defun common-lisp-hyperspec-glossary-pages ()
;;   "List of CLHS glossary pages."
;;   (mapcar (lambda (end)
;;	    (format "%sBody/26_glo_%s.htm"
;;		    common-lisp-hyperspec-root
;;		    end))
;;	  (cons "9" (mapcar #'char-to-string
;;			    (number-sequence ?a ?z)))))

;; (defun common-lisp-hyperspec-glossary-download ()
;;   "Download CLHS glossary pages to temporary files and return a
;; list of file names."
;;   (mapcar (lambda (url)
;;	    (url-file-local-copy url))
;;	  (common-lisp-hyperspec-glossary-pages)))

;; (defun common-lisp-hyperspec-glossary-entries (file)
;;   "Given a CLHS glossary file FILE, return a list of
;; term-anchor pairs.

;; Term is the glossary term and anchor is the term's anchor on the
;; page."
;;   (let (entries)
;;     (save-excursion
;;       (set-buffer (find-file-noselect file))
;;       (goto-char (point-min))
;;       (while (search-forward-regexp "<a\\ name=\"\\(.*?\\)\"><b>\\(.*?\\)</b>" nil t)
;;	(setq entries (cons (list (match-string-no-properties 2)
;;				  (match-string-no-properties 1))
;;			    entries))))
;;     (sort entries (lambda (a b)
;;		    (string< (car a) (car b))))))

;; ;; Add glossary terms by downloading and parsing glossary pages from CLHS
;; (mapc (lambda (entry)
;;        (puthash (car entry) (cadr entry)
;;		common-lisp-hyperspec--glossary-terms))
;;      (cl-reduce (lambda (a b)
;;	     (append a b))
;;	   (mapcar #'common-lisp-hyperspec-glossary-entries
;;		   (common-lisp-hyperspec-glossary-download))))

;; Add glossary entries to the master hash table
(mapc (lambda (entry)
	(puthash (car entry) (cadr entry)
		 common-lisp-hyperspec--glossary-terms))
      '(("()" "OPCP")
	("absolute" "absolute")
	("access" "access")
	("accessibility" "accessibility")
	("accessible" "accessible")
	("accessor" "accessor")
	("active" "active")
	("actual adjustability" "actual_adjustability")
	("actual argument" "actual_argument")
	("actual array element type" "actual_array_element_type")
	("actual complex part type" "actual_complex_part_type")
	("actual parameter" "actual_parameter")
	("actually adjustable" "actually_adjustable")
	("adjustability" "adjustability")
	("adjustable" "adjustable")
	("after method" "after_method")
	("alist" "alist")
	("alphabetic" "alphabetic")
	("alphanumeric" "alphanumeric")
	("ampersand" "ampersand")
	("anonymous" "anonymous")
	("apparently uninterned" "apparently_uninterned")
	("applicable" "applicable")
	("applicable handler" "applicable_handler")
	("applicable method" "applicable_method")
	("applicable restart" "applicable_restart")
	("apply" "apply")
	("argument" "argument")
	("argument evaluation order" "argument_evaluation_order")
	("argument precedence order" "argument_precedence_order")
	("around method" "around_method")
	("array" "array")
	("array element type" "array_element_type")
	("array total size" "array_total_size")
	("assign" "assign")
	("association list" "association_list")
	("asterisk" "asterisk")
	("at-sign" "at-sign")
	("atom" "atom")
	("atomic" "atomic")
	("atomic type specifier" "atomic_type_specifier")
	("attribute" "attribute")
	("aux variable" "aux_variable")
	("auxiliary method" "auxiliary_method")
	("backquote" "backquote")
	("backslash" "backslash")
	("base character" "base_character")
	("base string" "base_string")
	("before method" "before_method")
	("bidirectional" "bidirectional")
	("binary" "binary")
	("bind" "bind")
	("binding" "binding")
	("bit" "bit")
	("bit array" "bit_array")
	("bit vector" "bit_vector")
	("bit-wise logical operation specifier" "bit-wise_logical_operation_specifier")
	("block" "block")
	("block tag" "block_tag")
	("boa lambda list" "boa_lambda_list")
	("body parameter" "body_parameter")
	("boolean" "boolean")
	("boolean equivalent" "boolean_equivalent")
	("bound" "bound")
	("bound declaration" "bound_declaration")
	("bounded" "bounded")
	("bounding index" "bounding_index")
	("bounding index designator" "bounding_index_designator")
	("break loop" "break_loop")
	("broadcast stream" "broadcast_stream")
	("built-in class" "built-in_class")
	("built-in type" "built-in_type")
	("byte" "byte")
	("byte specifier" "byte_specifier")
	("cadr" "cadr")
	("call" "call")
	("captured initialization form" "captured_initialization_form")
	("car" "car")
	("case" "case")
	("case sensitivity mode" "case_sensitivity_mode")
	("catch" "catch")
	("catch tag" "catch_tag")
	("cddr" "cddr")
	("cdr" "cdr")
	("cell" "cell")
	("character" "character")
	("character code" "character_code")
	("character designator" "character_designator")
	("circular" "circular")
	("circular list" "circular_list")
	("class" "class")
	("class designator" "class_designator")
	("class precedence list" "class_precedence_list")
	("close" "close")
	("closed" "closed")
	("closure" "closure")
	("coalesce" "coalesce")
	("code" "code")
	("coerce" "coerce")
	("colon" "colon")
	("comma" "comma")
	("compilation" "compilation")
	("compilation environment" "compilation_environment")
	("compilation unit" "compilation_unit")
	("compile" "compile")
	("compile time" "compile_time")
	("compile-time definition" "compile-time_definition")
	("compiled code" "compiled_code")
	("compiled file" "compiled_file")
	("compiled function" "compiled_function")
	("compiler" "compiler")
	("compiler macro" "compiler_macro")
	("compiler macro expansion" "compiler_macro_expansion")
	("compiler macro form" "compiler_macro_form")
	("compiler macro function" "compiler_macro_function")
	("complex" "complex")
	("complex float" "complex_float")
	("complex part type" "complex_part_type")
	("complex rational" "complex_rational")
	("complex single float" "complex_single_float")
	("composite stream" "composite_stream")
	("compound form" "compound_form")
	("compound type specifier" "compound_type_specifier")
	("concatenated stream" "concatenated_stream")
	("condition" "condition")
	("condition designator" "condition_designator")
	("condition handler" "condition_handler")
	("condition reporter" "condition_reporter")
	("conditional newline" "conditional_newline")
	("conformance" "conformance")
	("conforming code" "conforming_code")
	("conforming implementation" "conforming_implementation")
	("conforming processor" "conforming_processor")
	("conforming program" "conforming_program")
	("congruent" "congruent")
	("cons" "cons")
	("constant" "constant")
	("constant form" "constant_form")
	("constant object" "constant_object")
	("constant variable" "constant_variable")
	("constituent" "constituent")
	("constituent trait" "constituent_trait")
	("constructed stream" "constructed_stream")
	("contagion" "contagion")
	("continuable" "continuable")
	("control form" "control_form")
	("copy" "copy")
	("correctable" "correctable")
	("current input base" "current_input_base")
	("current logical block" "current_logical_block")
	("current output base" "current_output_base")
	("current package" "current_package")
	("current pprint dispatch table" "current_pprint_dispatch_table")
	("current random state" "current_random_state")
	("current readtable" "current_readtable")
	("data type" "data_type")
	("debug I/O" "debug_iSLo")
	("debugger" "debugger")
	("declaration" "declaration")
	("declaration identifier" "declaration_identifier")
	("declaration specifier" "declaration_specifier")
	("declare" "declare")
	("decline" "decline")
	("decoded time" "decoded_time")
	("default method" "default_method")
	("defaulted initialization argument list" "defaulted_initialization_argument_list")
	("define-method-combination arguments lambda list" "define-method-combination_arguments_lambda_list")
	("define-modify-macro lambda list" "define-modify-macro_lambda_list")
	("defined name" "defined_name")
	("defining form" "defining_form")
	("defsetf lambda list" "defsetf_lambda_list")
	("deftype lambda list" "deftype_lambda_list")
	("denormalized" "denormalized")
	("derived type" "derived_type")
	("derived type specifier" "derived_type_specifier")
	("designator" "designator")
	("destructive" "destructive")
	("destructuring lambda list" "destructuring_lambda_list")
	("different" "different")
	("digit" "digit")
	("dimension" "dimension")
	("direct instance" "direct_instance")
	("direct subclass" "direct_subclass")
	("direct superclass" "direct_superclass")
	("disestablish" "disestablish")
	("disjoint" "disjoint")
	("dispatching macro character" "dispatching_macro_character")
	("displaced array" "displaced_array")
	("distinct" "distinct")
	("documentation string" "documentation_string")
	("dot" "dot")
	("dotted list" "dotted_list")
	("dotted pair" "dotted_pair")
	("double float" "double_float")
	("double-quote" "double-quote")
	("dynamic binding" "dynamic_binding")
	("dynamic environment" "dynamic_environment")
	("dynamic extent" "dynamic_extent")
	("dynamic scope" "dynamic_scope")
	("dynamic variable" "dynamic_variable")
	("echo stream" "echo_stream")
	("effective method" "effective_method")
	("element" "element")
	("element type" "element_type")
	("em" "em")
	("empty list" "empty_list")
	("empty type" "empty_type")
	("end of file" "end_of_file")
	("environment" "environment")
	("environment object" "environment_object")
	("environment parameter" "environment_parameter")
	("error" "error")
	("error output" "error_output")
	("escape" "escape")
	("establish" "establish")
	("evaluate" "evaluate")
	("evaluation" "evaluation")
	("evaluation environment" "evaluation_environment")
	("execute" "execute")
	("execution time" "execution_time")
	("exhaustive partition" "exhaustive_partition")
	("exhaustive union" "exhaustive_union")
	("exit point" "exit_point")
	("explicit return" "explicit_return")
	("explicit use" "explicit_use")
	("exponent marker" "exponent_marker")
	("export" "export")
	("exported" "exported")
	("expressed adjustability" "expressed_adjustability")
	("expressed array element type" "expressed_array_element_type")
	("expressed complex part type" "expressed_complex_part_type")
	("expression" "expression")
	("expressly adjustable" "expressly_adjustable")
	("extended character" "extended_character")
	("extended function designator" "extended_function_designator")
	("extended lambda list" "extended_lambda_list")
	("extension" "extension")
	("extent" "extent")
	("external file format" "external_file_format")
	("external file format designator" "external_file_format_designator")
	("external symbol" "external_symbol")
	("externalizable object" "externalizable_object")
	("false" "false")
	("fbound" "fbound")
	("feature" "feature")
	("feature expression" "feature_expression")
	("features list" "features_list")
	("file" "file")
	("file compiler" "file_compiler")
	("file position" "file_position")
	("file position designator" "file_position_designator")
	("file stream" "file_stream")
	("file system" "file_system")
	("filename" "filename")
	("fill pointer" "fill_pointer")
	("finite" "finite")
	("fixnum" "fixnum")
	("float" "float")
	("for-value" "for-value")
	("form" "form")
	("formal argument" "formal_argument")
	("formal parameter" "formal_parameter")
	("format" "format")
	("format argument" "format_argument")
	("format control" "format_control")
	("format directive" "format_directive")
	("format string" "format_string")
	("free declaration" "free_declaration")
	("fresh" "fresh")
	("freshline" "freshline")
	("funbound" "funbound")
	("function" "function")
	("function block name" "function_block_name")
	("function cell" "function_cell")
	("function designator" "function_designator")
	("function form" "function_form")
	("function name" "function_name")
	("functional evaluation" "functional_evaluation")
	("functional value" "functional_value")
	("further compilation" "further_compilation")
	("general" "general")
	("generalized boolean" "generalized_boolean")
	("generalized instance" "generalized_instance")
	("generalized reference" "generalized_reference")
	("generalized synonym stream" "generalized_synonym_stream")
	("generic function" "generic_function")
	("generic function lambda list" "generic_function_lambda_list")
	("gensym" "gensym")
	("global declaration" "global_declaration")
	("global environment" "global_environment")
	("global variable" "global_variable")
	("glyph" "glyph")
	("go" "go")
	("go point" "go_point")
	("go tag" "go_tag")
	("graphic" "graphic")
	("handle" "handle")
	("handler" "handler")
	("hash table" "hash_table")
	("home package" "home_package")
	("I/O customization variable" "iSLo_customization_variable")
	("identical" "identical")
	("identifier" "identifier")
	("immutable" "immutable")
	("implementation" "implementation")
	("implementation limit" "implementation_limit")
	("implementation-defined" "implementation-defined")
	("implementation-dependent" "implementation-dependent")
	("implementation-independent" "implementation-independent")
	("implicit block" "implicit_block")
	("implicit compilation" "implicit_compilation")
	("implicit progn" "implicit_progn")
	("implicit tagbody" "implicit_tagbody")
	("import" "import")
	("improper list" "improper_list")
	("inaccessible" "inaccessible")
	("indefinite extent" "indefinite_extent")
	("indefinite scope" "indefinite_scope")
	("indicator" "indicator")
	("indirect instance" "indirect_instance")
	("inherit" "inherit")
	("initial pprint dispatch table" "initial_pprint_dispatch_table")
	("initial readtable" "initial_readtable")
	("initialization argument list" "initialization_argument_list")
	("initialization form" "initialization_form")
	("input" "input")
	("instance" "instance")
	("integer" "integer")
	("interactive stream" "interactive_stream")
	("intern" "intern")
	("internal symbol" "internal_symbol")
	("internal time" "internal_time")
	("internal time unit" "internal_time_unit")
	("interned" "interned")
	("interpreted function" "interpreted_function")
	("interpreted implementation" "interpreted_implementation")
	("interval designator" "interval_designator")
	("invalid" "invalid")
	("iteration form" "iteration_form")
	("iteration variable" "iteration_variable")
	("key" "key")
	("keyword" "keyword")
	("keyword parameter" "keyword_parameter")
	("keyword/value pair" "keywordSLvalue_pair")
	("Lisp image" "lisp_image")
	("Lisp printer" "lisp_printer")
	("Lisp read-eval-print loop" "lisp_read-eval-print_loop")
	("Lisp reader" "lisp_reader")
	("lambda combination" "lambda_combination")
	("lambda expression" "lambda_expression")
	("lambda form" "lambda_form")
	("lambda list" "lambda_list")
	("lambda list keyword" "lambda_list_keyword")
	("lambda variable" "lambda_variable")
	("leaf" "leaf")
	("leap seconds" "leap_seconds")
	("left-parenthesis" "left-parenthesis")
	("length" "length")
	("lexical binding" "lexical_binding")
	("lexical closure" "lexical_closure")
	("lexical environment" "lexical_environment")
	("lexical scope" "lexical_scope")
	("lexical variable" "lexical_variable")
	("list" "list")
	("list designator" "list_designator")
	("list structure" "list_structure")
	("literal" "literal")
	("load" "load")
	("load time" "load_time")
	("load time value" "load_time_value")
	("loader" "loader")
	("local declaration" "local_declaration")
	("local precedence order" "local_precedence_order")
	("local slot" "local_slot")
	("logical block" "logical_block")
	("logical host" "logical_host")
	("logical host designator" "logical_host_designator")
	("logical pathname" "logical_pathname")
	("long float" "long_float")
	("loop keyword" "loop_keyword")
	("lowercase" "lowercase")
	("Metaobject Protocol" "metaobject_protocol")
	("macro" "macro")
	("macro character" "macro_character")
	("macro expansion" "macro_expansion")
	("macro form" "macro_form")
	("macro function" "macro_function")
	("macro lambda list" "macro_lambda_list")
	("macro name" "macro_name")
	("macroexpand hook" "macroexpand_hook")
	("mapping" "mapping")
	("metaclass" "metaclass")
	("method" "method")
	("method combination" "method_combination")
	("method-defining form" "method-defining_form")
	("method-defining operator" "method-defining_operator")
	("minimal compilation" "minimal_compilation")
	("modified lambda list" "modified_lambda_list")
	("most recent" "most_recent")
	("multiple escape" "multiple_escape")
	("multiple values" "multiple_values")
	("name" "name")
	("named constant" "named_constant")
	("namespace" "namespace")
	("namestring" "namestring")
	("newline" "newline")
	("next method" "next_method")
	("nickname" "nickname")
	("nil" "nil")
	("non-atomic" "non-atomic")
	("non-constant variable" "non-constant_variable")
	("non-correctable" "non-correctable")
	("non-empty" "non-empty")
	("non-generic function" "non-generic_function")
	("non-graphic" "non-graphic")
	("non-list" "non-list")
	("non-local exit" "non-local_exit")
	("non-nil" "non-nil")
	("non-null lexical environment" "non-null_lexical_environment")
	("non-simple" "non-simple")
	("non-terminating" "non-terminating")
	("non-top-level form" "non-top-level_form")
	("normal return" "normal_return")
	("normalized" "normalized")
	("null" "null")
	("null lexical environment" "null_lexical_environment")
	("number" "number")
	("numeric" "numeric")
	("object" "object")
	("object-traversing" "object-traversing")
	("open" "open")
	("operator" "operator")
	("optimize quality" "optimize_quality")
	("optional parameter" "optional_parameter")
	("ordinary function" "ordinary_function")
	("ordinary lambda list" "ordinary_lambda_list")
	("otherwise inaccessible part" "otherwise_inaccessible_part")
	("output" "output")
	("package" "package")
	("package cell" "package_cell")
	("package designator" "package_designator")
	("package marker" "package_marker")
	("package prefix" "package_prefix")
	("package registry" "package_registry")
	("pairwise" "pairwise")
	("parallel" "parallel")
	("parameter" "parameter")
	("parameter specializer" "parameter_specializer")
	("parameter specializer name" "parameter_specializer_name")
	("pathname" "pathname")
	("pathname designator" "pathname_designator")
	("physical pathname" "physical_pathname")
	("place" "place")
	("plist" "plist")
	("portable" "portable")
	("potential copy" "potential_copy")
	("potential number" "potential_number")
	("pprint dispatch table" "pprint_dispatch_table")
	("predicate" "predicate")
	("present" "present")
	("pretty print" "pretty_print")
	("pretty printer" "pretty_printer")
	("pretty printing stream" "pretty_printing_stream")
	("primary method" "primary_method")
	("primary value" "primary_value")
	("principal" "principal")
	("print name" "print_name")
	("printer control variable" "printer_control_variable")
	("printer escaping" "printer_escaping")
	("printing" "printing")
	("process" "process")
	("processor" "processor")
	("proclaim" "proclaim")
	("proclamation" "proclamation")
	("prog tag" "prog_tag")
	("program" "program")
	("programmer" "programmer")
	("programmer code" "programmer_code")
	("proper list" "proper_list")
	("proper name" "proper_name")
	("proper sequence" "proper_sequence")
	("proper subtype" "proper_subtype")
	("property" "property")
	("property indicator" "property_indicator")
	("property list" "property_list")
	("property value" "property_value")
	("purports to conform" "purports_to_conform")
	("qualified method" "qualified_method")
	("qualifier" "qualifier")
	("query I/O" "query_iSLo")
	("quoted object" "quoted_object")
	("radix" "radix")
	("random state" "random_state")
	("rank" "rank")
	("ratio" "ratio")
	("ratio marker" "ratio_marker")
	("rational" "rational")
	("read" "read")
	("readably" "readably")
	("reader" "reader")
	("reader macro" "reader_macro")
	("reader macro function" "reader_macro_function")
	("readtable" "readtable")
	("readtable case" "readtable_case")
	("readtable designator" "readtable_designator")
	("recognizable subtype" "recognizable_subtype")
	("reference" "reference")
	("registered package" "registered_package")
	("relative" "relative")
	("repertoire" "repertoire")
	("report" "report")
	("report message" "report_message")
	("required parameter" "required_parameter")
	("rest list" "rest_list")
	("rest parameter" "rest_parameter")
	("restart" "restart")
	("restart designator" "restart_designator")
	("restart function" "restart_function")
	("return" "return")
	("return value" "return_value")
	("right-parenthesis" "right-parenthesis")
	("run time" "run_time")
	("run-time compiler" "run-time_compiler")
	("run-time definition" "run-time_definition")
	("run-time environment" "run-time_environment")
	("safe" "safe")
	("safe call" "safe_call")
	("same" "same")
	("satisfy the test" "satisfy_the_test")
	("scope" "scope")
	("script" "script")
	("secondary value" "secondary_value")
	("section" "section")
	("self-evaluating object" "self-evaluating_object")
	("semi-standard" "semi-standard")
	("semicolon" "semicolon")
	("sequence" "sequence")
	("sequence function" "sequence_function")
	("sequential" "sequential")
	("sequentially" "sequentially")
	("serious condition" "serious_condition")
	("session" "session")
	("set" "set")
	("setf expander" "setf_expander")
	("setf expansion" "setf_expansion")
	("setf function" "setf_function")
	("setf function name" "setf_function_name")
	("shadow" "shadow")
	("shadowing symbol" "shadowing_symbol")
	("shadowing symbols list" "shadowing_symbols_list")
	("shared slot" "shared_slot")
	("sharpsign" "sharpsign")
	("short float" "short_float")
	("sign" "sign")
	("signal" "signal")
	("signature" "signature")
	("similar" "similar")
	("similarity" "similarity")
	("simple" "simple")
	("simple array" "simple_array")
	("simple bit array" "simple_bit_array")
	("simple bit vector" "simple_bit_vector")
	("simple condition" "simple_condition")
	("simple general vector" "simple_general_vector")
	("simple string" "simple_string")
	("simple vector" "simple_vector")
	("single escape" "single_escape")
	("single float" "single_float")
	("single-quote" "single-quote")
	("singleton" "singleton")
	("situation" "situation")
	("slash" "slash")
	("slot" "slot")
	("slot specifier" "slot_specifier")
	("source code" "source_code")
	("source file" "source_file")
	("space" "space")
	("special form" "special_form")
	("special operator" "special_operator")
	("special variable" "special_variable")
	("specialize" "specialize")
	("specialized" "specialized")
	("specialized lambda list" "specialized_lambda_list")
	("spreadable argument list designator" "spreadable_argument_list_designator")
	("stack allocate" "stack_allocate")
	("stack-allocated" "stack-allocated")
	("standard character" "standard_character")
	("standard class" "standard_class")
	("standard generic function" "standard_generic_function")
	("standard input" "standard_input")
	("standard method combination" "standard_method_combination")
	("standard object" "standard_object")
	("standard output" "standard_output")
	("standard pprint dispatch table" "standard_pprint_dispatch_table")
	("standard readtable" "standard_readtable")
	("standard syntax" "standard_syntax")
	("standardized" "standardized")
	("startup environment" "startup_environment")
	("step" "step")
	("stream" "stream")
	("stream associated with a file" "stream_associated_with_a_file")
	("stream designator" "stream_designator")
	("stream element type" "stream_element_type")
	("stream variable" "stream_variable")
	("stream variable designator" "stream_variable_designator")
	("string" "string")
	("string designator" "string_designator")
	("string equal" "string_equal")
	("string stream" "string_stream")
	("structure" "structure")
	("structure class" "structure_class")
	("structure name" "structure_name")
	("style warning" "style_warning")
	("subclass" "subclass")
	("subexpression" "subexpression")
	("subform" "subform")
	("subrepertoire" "subrepertoire")
	("subtype" "subtype")
	("superclass" "superclass")
	("supertype" "supertype")
	("supplied-p parameter" "supplied-p_parameter")
	("symbol" "symbol")
	("symbol macro" "symbol_macro")
	("synonym stream" "synonym_stream")
	("synonym stream symbol" "synonym_stream_symbol")
	("syntax type" "syntax_type")
	("system class" "system_class")
	("system code" "system_code")
	("t" "t")
	("tag" "tag")
	("tail" "tail")
	("target" "target")
	("terminal I/O" "terminal_iSLo")
	("terminating" "terminating")
	("tertiary value" "tertiary_value")
	("throw" "throw")
	("tilde" "tilde")
	("time" "time")
	("time zone" "time_zone")
	("token" "token")
	("top level form" "top_level_form")
	("trace output" "trace_output")
	("tree" "tree")
	("tree structure" "tree_structure")
	("true" "true")
	("truename" "truename")
	("two-way stream" "two-way_stream")
	("type" "type")
	("type declaration" "type_declaration")
	("type equivalent" "type_equivalent")
	("type expand" "type_expand")
	("type specifier" "type_specifier")
	("unbound" "unbound")
	("unbound variable" "unbound_variable")
	("undefined function" "undefined_function")
	("unintern" "unintern")
	("uninterned" "uninterned")
	("universal time" "universal_time")
	("unqualified method" "unqualified_method")
	("unregistered package" "unregistered_package")
	("unsafe" "unsafe")
	("unsafe call" "unsafe_call")
	("upgrade" "upgrade")
	("upgraded array element type" "upgraded_array_element_type")
	("upgraded complex part type" "upgraded_complex_part_type")
	("uppercase" "uppercase")
	("use" "use")
	("use list" "use_list")
	("user" "user")
	("valid array dimension" "valid_array_dimension")
	("valid array index" "valid_array_index")
	("valid array row-major index" "valid_array_row-major_index")
	("valid fill pointer" "valid_fill_pointer")
	("valid logical pathname host" "valid_logical_pathname_host")
	("valid pathname device" "valid_pathname_device")
	("valid pathname directory" "valid_pathname_directory")
	("valid pathname host" "valid_pathname_host")
	("valid pathname name" "valid_pathname_name")
	("valid pathname type" "valid_pathname_type")
	("valid pathname version" "valid_pathname_version")
	("valid physical pathname host" "valid_physical_pathname_host")
	("valid sequence index" "valid_sequence_index")
	("value" "value")
	("value cell" "value_cell")
	("variable" "variable")
	("vector" "vector")
	("vertical-bar" "vertical-bar")
	("whitespace" "whitespace")
	("wild" "wild")
	("write" "write")
	("writer" "writer")
	("yield" "yield")))

(defun common-lisp-hyperspec-glossary-term (term)
  "View the definition of TERM on the Common Lisp Hyperspec."
  (interactive
   (list
    (completing-read "Look up glossary term: "
		     common-lisp-hyperspec--glossary-terms nil t)))
  (browse-url (funcall common-lisp-hyperspec-glossary-function term)))

(defun common-lisp-glossary-6.0 (term)
  "Get a URL for a glossary term TERM."
  (let ((anchor (gethash term common-lisp-hyperspec--glossary-terms)))
    (if (not anchor)
	(message "Unknown glossary term: %s" term)
      (format "%sBody/26_glo_%s.htm#%s"
	      common-lisp-hyperspec-root
	      (let ((char (string-to-char term)))
		(if (and (<= ?a char)
			 (<= char ?z))
		    (make-string 1 char)
		  "9"))
	      anchor))))

;; Tianxiang Xiong 20151229
;; Is this function necessary? The link does created does not work.
(defun common-lisp-glossary-4.0 (string)
  (format "%sBody/glo_%s.html#%s"
	  common-lisp-hyperspec-root
	  (let ((char (string-to-char string)))
	    (if (and (<= ?a char)
		     (<= char ?z))
		(make-string 1 char)
	      "9"))
	  (subst-char-in-string ?\  ?_ string)))


;;;; Issuex

;; FIXME: the issuex stuff is not used
(defvar common-lisp-hyperspec-issuex-table nil
  "The HyperSpec IssueX table file.  If you copy the HyperSpec to your
local system, set this variable to the location of the Issue
cross-references table which is usually \"Map_IssX.txt\" or
\"Issue-Cross-Refs.text\".")

(defvar common-lisp-hyperspec--issuex-symbols
  (make-hash-table :test 'equal))

(mapc
 (lambda (entry)
   (puthash (car entry) (cadr entry) common-lisp-hyperspec--issuex-symbols))
 (if common-lisp-hyperspec-issuex-table
     (common-lisp-hyperspec--parse-map-file
      common-lisp-hyperspec-issuex-table)
   '(("&environment-binding-order:first" "iss001.htm")
     ("access-error-name" "iss002.htm")
     ("adjust-array-displacement" "iss003.htm")
     ("adjust-array-fill-pointer" "iss004.htm")
     ("adjust-array-not-adjustable:implicit-copy" "iss005.htm")
     ("allocate-instance:add" "iss006.htm")
     ("allow-local-inline:inline-notinline" "iss007.htm")
     ("allow-other-keys-nil:permit" "iss008.htm")
     ("aref-1d" "iss009.htm")
     ("argument-mismatch-error-again:consistent" "iss010.htm")
     ("argument-mismatch-error-moon:fix" "iss011.htm")
     ("argument-mismatch-error:more-clarifications" "iss012.htm")
     ("arguments-underspecified:specify" "iss013.htm")
     ("array-dimension-limit-implications:all-fixnum" "iss014.htm")
     ("array-type-element-type-semantics:unify-upgrading" "iss015.htm")
     ("assert-error-type:error" "iss016.htm")
     ("assoc-rassoc-if-key" "iss017.htm")
     ("assoc-rassoc-if-key:yes" "iss018.htm")
     ("boa-aux-initialization:error-on-read" "iss019.htm")
     ("break-on-warnings-obsolete:remove" "iss020.htm")
     ("broadcast-stream-return-values:clarify-minimally" "iss021.htm")
     ("butlast-negative:should-signal" "iss022.htm")
     ("change-class-initargs:permit" "iss023.htm")
     ("char-name-case:x3j13-mar-91" "iss024.htm")
     ("character-loose-ends:fix" "iss025.htm")
     ("character-proposal:2" "iss026.htm")
     ("character-proposal:2-1-1" "iss027.htm")
     ("character-proposal:2-1-2" "iss028.htm")
     ("character-proposal:2-2-1" "iss029.htm")
     ("character-proposal:2-3-1" "iss030.htm")
     ("character-proposal:2-3-2" "iss031.htm")
     ("character-proposal:2-3-3" "iss032.htm")
     ("character-proposal:2-3-4" "iss033.htm")
     ("character-proposal:2-3-5" "iss034.htm")
     ("character-proposal:2-3-6" "iss035.htm")
     ("character-proposal:2-4-1" "iss036.htm")
     ("character-proposal:2-4-2" "iss037.htm")
     ("character-proposal:2-4-3" "iss038.htm")
     ("character-proposal:2-5-2" "iss039.htm")
     ("character-proposal:2-5-6" "iss040.htm")
     ("character-proposal:2-5-7" "iss041.htm")
     ("character-proposal:2-6-1" "iss042.htm")
     ("character-proposal:2-6-2" "iss043.htm")
     ("character-proposal:2-6-3" "iss044.htm")
     ("character-proposal:2-6-5" "iss045.htm")
     ("character-vs-char:less-inconsistent-short" "iss046.htm")
     ("class-object-specializer:affirm" "iss047.htm")
     ("clos-conditions-again:allow-subset" "iss048.htm")
     ("clos-conditions:integrate" "iss049.htm")
     ("clos-error-checking-order:no-applicable-method-first" "iss050.htm")
     ("clos-macro-compilation:minimal" "iss051.htm")
     ("close-constructed-stream:argument-stream-only" "iss052.htm")
     ("closed-stream-operations:allow-inquiry" "iss053.htm")
     ("coercing-setf-name-to-function:all-function-names" "iss054.htm")
     ("colon-number" "iss055.htm")
     ("common-features:specify" "iss056.htm")
     ("common-type:remove" "iss057.htm")
     ("compile-argument-problems-again:fix" "iss058.htm")
     ("compile-file-handling-of-top-level-forms:clarify" "iss059.htm")
     ("compile-file-output-file-defaults:input-file" "iss060.htm")
     ("compile-file-package" "iss061.htm")
     ("compile-file-pathname-arguments:make-consistent" "iss062.htm")
     ("compile-file-symbol-handling:new-require-consistency" "iss063.htm")
     ("compiled-function-requirements:tighten" "iss064.htm")
     ("compiler-diagnostics:use-handler" "iss065.htm")
     ("compiler-let-confusion:eliminate" "iss066.htm")
     ("compiler-verbosity:like-load" "iss067.htm")
     ("compiler-warning-stream" "iss068.htm")
     ("complex-atan-branch-cut:tweak" "iss069.htm")
     ("complex-atanh-bogus-formula:tweak-more" "iss070.htm")
     ("complex-rational-result:extend" "iss071.htm")
     ("compute-applicable-methods:generic" "iss072.htm")
     ("concatenate-sequence:signal-error" "iss073.htm")
     ("condition-accessors-setfable:no" "iss074.htm")
     ("condition-restarts:buggy" "iss075.htm")
     ("condition-restarts:permit-association" "iss076.htm")
     ("condition-slots:hidden" "iss077.htm")
     ("cons-type-specifier:add" "iss078.htm")
     ("constant-circular-compilation:yes" "iss079.htm")
     ("constant-collapsing:generalize" "iss080.htm")
     ("constant-compilable-types:specify" "iss081.htm")
     ("constant-function-compilation:no" "iss082.htm")
     ("constant-modification:disallow" "iss083.htm")
     ("constantp-definition:intentional" "iss084.htm")
     ("constantp-environment:add-arg" "iss085.htm")
     ("contagion-on-numerical-comparisons:transitive" "iss086.htm")
     ("copy-symbol-copy-plist:copy-list" "iss087.htm")
     ("copy-symbol-print-name:equal" "iss088.htm")
     ("data-io:add-support" "iss089.htm")
     ("data-types-hierarchy-underspecified" "iss090.htm")
     ("debugger-hook-vs-break:clarify" "iss091.htm")
     ("declaration-scope:no-hoisting" "iss092.htm")
     ("declare-array-type-element-references:restrictive" "iss093.htm")
     ("declare-function-ambiguity:delete-ftype-abbreviation" "iss094.htm")
     ("declare-macros:flush" "iss095.htm")
     ("declare-type-free:lexical" "iss096.htm")
     ("decls-and-doc" "iss097.htm")
     ("decode-universal-time-daylight:like-encode" "iss098.htm")
     ("defconstant-special:no" "iss099.htm")
     ("defgeneric-declare:allow-multiple" "iss100.htm")
     ("define-compiler-macro:x3j13-nov89" "iss101.htm")
     ("define-condition-syntax:\
incompatibly-more-like-defclass+emphasize-read-only" "iss102.htm")
     ("define-method-combination-behavior:clarify" "iss103.htm")
     ("defining-macros-non-top-level:allow" "iss104.htm")
     ("defmacro-block-scope:excludes-bindings" "iss105.htm")
     ("defmacro-lambda-list:tighten-description" "iss106.htm")
     ("defmethod-declaration-scope:corresponds-to-bindings" "iss107.htm")
     ("defpackage:addition" "iss108.htm")
     ("defstruct-constructor-key-mixture:allow-key" "iss109.htm")
     ("defstruct-constructor-options:explicit" "iss110.htm")
     ("defstruct-constructor-slot-variables:not-bound" "iss111.htm")
     ("defstruct-copier-argument-type:restrict" "iss112.htm")
     ("defstruct-copier:argument-type" "iss113.htm")
     ("defstruct-default-value-evaluation:iff-needed" "iss114.htm")
     ("defstruct-include-deftype:explicitly-undefined" "iss115.htm")
     ("defstruct-print-function-again:x3j13-mar-93" "iss116.htm")
     ("defstruct-print-function-inheritance:yes" "iss117.htm")
     ("defstruct-redefinition:error" "iss118.htm")
     ("defstruct-slots-constraints-name:duplicates-error" "iss119.htm")
     ("defstruct-slots-constraints-number" "iss120.htm")
     ("deftype-destructuring:yes" "iss121.htm")
     ("deftype-key:allow" "iss122.htm")
     ("defvar-documentation:unevaluated" "iss123.htm")
     ("defvar-init-time:not-delayed" "iss124.htm")
     ("defvar-initialization:conservative" "iss125.htm")
     ("deprecation-position:limited" "iss126.htm")
     ("describe-interactive:no" "iss127.htm")
     ("describe-underspecified:describe-object" "iss128.htm")
     ("destructive-operations:specify" "iss129.htm")
     ("destructuring-bind:new-macro" "iss130.htm")
     ("disassemble-side-effect:do-not-install" "iss131.htm")
     ("displaced-array-predicate:add" "iss132.htm")
     ("do-symbols-block-scope:entire-form" "iss133.htm")
     ("do-symbols-duplicates" "iss134.htm")
     ("documentation-function-bugs:fix" "iss135.htm")
     ("documentation-function-tangled:require-argument" "iss136.htm")
     ("dotimes-ignore:x3j13-mar91" "iss137.htm")
     ("dotted-list-arguments:clarify" "iss138.htm")
     ("dotted-macro-forms:allow" "iss139.htm")
     ("dribble-technique" "iss140.htm")
     ("dynamic-extent-function:extend" "iss141.htm")
     ("dynamic-extent:new-declaration" "iss142.htm")
     ("equal-structure:maybe-status-quo" "iss143.htm")
     ("error-terminology-warning:might" "iss144.htm")
     ("eval-other:self-evaluate" "iss145.htm")
     ("eval-top-level:load-like-compile-file" "iss146.htm")
     ("eval-when-non-top-level:generalize-eval-new-keywords" "iss147.htm")
     ("eval-when-obsolete-keywords:x3j13-mar-1993" "iss148.htm")
     ("evalhook-step-confusion:fix" "iss149.htm")
     ("evalhook-step-confusion:x3j13-nov-89" "iss150.htm")
     ("exit-extent-and-condition-system:like-dynamic-bindings" "iss151.htm")
     ("exit-extent:minimal" "iss152.htm")
     ("expt-ratio:p.211" "iss153.htm")
     ("extensions-position:documentation" "iss154.htm")
     ("external-format-for-every-file-connection:minimum" "iss155.htm")
     ("extra-return-values:no" "iss156.htm")
     ("file-open-error:signal-file-error" "iss157.htm")
     ("fixnum-non-portable:tighten-definition" "iss158.htm")
     ("flet-declarations" "iss159.htm")
     ("flet-declarations:allow" "iss160.htm")
     ("flet-implicit-block:yes" "iss161.htm")
     ("float-underflow:add-variables" "iss162.htm")
     ("floating-point-condition-names:x3j13-nov-89" "iss163.htm")
     ("format-atsign-colon" "iss164.htm")
     ("format-colon-uparrow-scope" "iss165.htm")
     ("format-comma-interval" "iss166.htm")
     ("format-e-exponent-sign:force-sign" "iss167.htm")
     ("format-op-c" "iss168.htm")
     ("format-pretty-print:yes" "iss169.htm")
     ("format-string-arguments:specify" "iss170.htm")
     ("function-call-evaluation-order:more-unspecified" "iss171.htm")
     ("function-composition:jan89-x3j13" "iss172.htm")
     ("function-definition:jan89-x3j13" "iss173.htm")
     ("function-name:large" "iss174.htm")
     ("function-type" "iss175.htm")
     ("function-type-argument-type-semantics:restrictive" "iss176.htm")
     ("function-type-key-name:specify-keyword" "iss177.htm")
     ("function-type-rest-list-element:use-actual-argument-type" "iss178.htm")
     ("function-type:x3j13-march-88" "iss179.htm")
     ("generalize-pretty-printer:unify" "iss180.htm")
     ("generic-flet-poorly-designed:delete" "iss181.htm")
     ("gensym-name-stickiness:like-teflon" "iss182.htm")
     ("gentemp-bad-idea:deprecate" "iss183.htm")
     ("get-macro-character-readtable:nil-standard" "iss184.htm")
     ("get-setf-method-environment:add-arg" "iss185.htm")
     ("hash-table-access:x3j13-mar-89" "iss186.htm")
     ("hash-table-key-modification:specify" "iss187.htm")
     ("hash-table-package-generators:add-with-wrapper" "iss188.htm")
     ("hash-table-rehash-size-integer" "iss189.htm")
     ("hash-table-size:intended-entries" "iss190.htm")
     ("hash-table-tests:add-equalp" "iss191.htm")
     ("ieee-atan-branch-cut:split" "iss192.htm")
     ("ignore-use-terminology:value-only" "iss193.htm")
     ("import-setf-symbol-package" "iss194.htm")
     ("in-package-functionality:mar89-x3j13" "iss195.htm")
     ("in-syntax:minimal" "iss196.htm")
     ("initialization-function-keyword-checking" "iss197.htm")
     ("iso-compatibility:add-substrate" "iss198.htm")
     ("jun90-trivial-issues:11" "iss199.htm")
     ("jun90-trivial-issues:14" "iss200.htm")
     ("jun90-trivial-issues:24" "iss201.htm")
     ("jun90-trivial-issues:25" "iss202.htm")
     ("jun90-trivial-issues:27" "iss203.htm")
     ("jun90-trivial-issues:3" "iss204.htm")
     ("jun90-trivial-issues:4" "iss205.htm")
     ("jun90-trivial-issues:5" "iss206.htm")
     ("jun90-trivial-issues:9" "iss207.htm")
     ("keyword-argument-name-package:any" "iss208.htm")
     ("last-n" "iss209.htm")
     ("lcm-no-arguments:1" "iss210.htm")
     ("lexical-construct-global-definition:undefined" "iss211.htm")
     ("lisp-package-name:common-lisp" "iss212.htm")
     ("lisp-symbol-redefinition-again:more-fixes" "iss213.htm")
     ("lisp-symbol-redefinition:mar89-x3j13" "iss214.htm")
     ("load-objects:make-load-form" "iss215.htm")
     ("load-time-eval:r**2-new-special-form" "iss216.htm")
     ("load-time-eval:r**3-new-special-form" "iss217.htm")
     ("load-truename:new-pathname-variables" "iss218.htm")
     ("locally-top-level:special-form" "iss219.htm")
     ("loop-and-discrepancy:no-reiteration" "iss220.htm")
     ("loop-for-as-on-typo:fix-typo" "iss221.htm")
     ("loop-initform-environment:partial-interleaving-vague" "iss222.htm")
     ("loop-miscellaneous-repairs:fix" "iss223.htm")
     ("loop-named-block-nil:override" "iss224.htm")
     ("loop-present-symbols-typo:flush-wrong-words" "iss225.htm")
     ("loop-syntax-overhaul:repair" "iss226.htm")
     ("macro-as-function:disallow" "iss227.htm")
     ("macro-declarations:make-explicit" "iss228.htm")
     ("macro-environment-extent:dynamic" "iss229.htm")
     ("macro-function-environment" "iss230.htm")
     ("macro-function-environment:yes" "iss231.htm")
     ("macro-subforms-top-level-p:add-constraints" "iss232.htm")
     ("macroexpand-hook-default:explicitly-vague" "iss233.htm")
     ("macroexpand-hook-initial-value:implementation-dependent" "iss234.htm")
     ("macroexpand-return-value:true" "iss235.htm")
     ("make-load-form-confusion:rewrite" "iss236.htm")
     ("make-load-form-saving-slots:no-initforms" "iss237.htm")
     ("make-package-use-default:implementation-dependent" "iss238.htm")
     ("map-into:add-function" "iss239.htm")
     ("mapping-destructive-interaction:explicitly-vague" "iss240.htm")
     ("metaclass-of-system-class:unspecified" "iss241.htm")
     ("method-combination-arguments:clarify" "iss242.htm")
     ("method-initform:forbid-call-next-method" "iss243.htm")
     ("muffle-warning-condition-argument" "iss244.htm")
     ("multiple-value-setq-order:like-setf-of-values" "iss245.htm")
     ("multiple-values-limit-on-variables:undefined" "iss246.htm")
     ("nintersection-destruction" "iss247.htm")
     ("nintersection-destruction:revert" "iss248.htm")
     ("not-and-null-return-value:x3j13-mar-93" "iss249.htm")
     ("nth-value:add" "iss250.htm")
     ("optimize-debug-info:new-quality" "iss251.htm")
     ("package-clutter:reduce" "iss252.htm")
     ("package-deletion:new-function" "iss253.htm")
     ("package-function-consistency:more-permissive" "iss254.htm")
     ("parse-error-stream:split-types" "iss255.htm")
     ("pathname-component-case:keyword-argument" "iss256.htm")
     ("pathname-component-value:specify" "iss257.htm")
     ("pathname-host-parsing:recognize-logical-host-names" "iss258.htm")
     ("pathname-logical:add" "iss259.htm")
     ("pathname-print-read:sharpsign-p" "iss260.htm")
     ("pathname-stream" "iss261.htm")
     ("pathname-stream:files-or-synonym" "iss262.htm")
     ("pathname-subdirectory-list:new-representation" "iss263.htm")
     ("pathname-symbol" "iss264.htm")
     ("pathname-syntax-error-time:explicitly-vague" "iss265.htm")
     ("pathname-unspecific-component:new-token" "iss266.htm")
     ("pathname-wild:new-functions" "iss267.htm")
     ("peek-char-read-char-echo:first-read-char" "iss268.htm")
     ("plist-duplicates:allow" "iss269.htm")
     ("pretty-print-interface" "iss270.htm")
     ("princ-readably:x3j13-dec-91" "iss271.htm")
     ("print-case-behavior:clarify" "iss272.htm")
     ("print-case-print-escape-interaction:vertical-bar-rule-no-upcase"
      "iss273.htm")
     ("print-circle-shared:respect-print-circle" "iss274.htm")
     ("print-circle-structure:user-functions-work" "iss275.htm")
     ("print-readably-behavior:clarify" "iss276.htm")
     ("printer-whitespace:just-one-space" "iss277.htm")
     ("proclaim-etc-in-compile-file:new-macro" "iss278.htm")
     ("push-evaluation-order:first-item" "iss279.htm")
     ("push-evaluation-order:item-first" "iss280.htm")
     ("pushnew-store-required:unspecified" "iss281.htm")
     ("quote-semantics:no-copying" "iss282.htm")
     ("range-of-count-keyword:nil-or-integer" "iss283.htm")
     ("range-of-start-and-end-parameters:integer-and-integer-nil" "iss284.htm")
     ("read-and-write-bytes:new-functions" "iss285.htm")
     ("read-case-sensitivity:readtable-keywords" "iss286.htm")
     ("read-modify-write-evaluation-order:delayed-access-stores" "iss287.htm")
     ("read-suppress-confusing:generalize" "iss288.htm")
     ("reader-error:new-type" "iss289.htm")
     ("real-number-type:x3j13-mar-89" "iss290.htm")
     ("recursive-deftype:explicitly-vague" "iss291.htm")
     ("reduce-argument-extraction" "iss292.htm")
     ("remf-destruction-unspecified:x3j13-mar-89" "iss293.htm")
     ("require-pathname-defaults-again:x3j13-dec-91" "iss294.htm")
     ("require-pathname-defaults-yet-again:restore-argument" "iss295.htm")
     ("require-pathname-defaults:eliminate" "iss296.htm")
     ("rest-list-allocation:may-share" "iss297.htm")
     ("result-lists-shared:specify" "iss298.htm")
     ("return-values-unspecified:specify" "iss299.htm")
     ("room-default-argument:new-value" "iss300.htm")
     ("self-modifying-code:forbid" "iss301.htm")
     ("sequence-type-length:must-match" "iss302.htm")
     ("setf-apply-expansion:ignore-expander" "iss303.htm")
     ("setf-find-class:allow-nil" "iss304.htm")
     ("setf-functions-again:minimal-changes" "iss305.htm")
     ("setf-get-default:evaluated-but-ignored" "iss306.htm")
     ("setf-macro-expansion:last" "iss307.htm")
     ("setf-method-vs-setf-method:rename-old-terms" "iss308.htm")
     ("setf-multiple-store-variables:allow" "iss309.htm")
     ("setf-of-apply:only-aref-and-friends" "iss310.htm")
     ("setf-of-values:add" "iss311.htm")
     ("setf-sub-methods:delayed-access-stores" "iss312.htm")
     ("shadow-already-present" "iss313.htm")
     ("shadow-already-present:works" "iss314.htm")
     ("sharp-comma-confusion:remove" "iss315.htm")
     ("sharp-o-foobar:consequences-undefined" "iss316.htm")
     ("sharp-star-delimiter:normal-delimiter" "iss317.htm")
     ("sharpsign-plus-minus-package:keyword" "iss318.htm")
     ("slot-missing-values:specify" "iss319.htm")
     ("slot-value-metaclasses:less-minimal" "iss320.htm")
     ("special-form-p-misnomer:rename" "iss321.htm")
     ("special-type-shadowing:clarify" "iss322.htm")
     ("standard-input-initial-binding:defined-contracts" "iss323.htm")
     ("standard-repertoire-gratuitous:rename" "iss324.htm")
     ("step-environment:current" "iss325.htm")
     ("step-minimal:permit-progn" "iss326.htm")
     ("stream-access:add-types-accessors" "iss327.htm")
     ("stream-capabilities:interactive-stream-p" "iss328.htm")
     ("string-coercion:make-consistent" "iss329.htm")
     ("string-output-stream-bashing:undefined" "iss330.htm")
     ("structure-read-print-syntax:keywords" "iss331.htm")
     ("subseq-out-of-bounds" "iss332.htm")
     ("subseq-out-of-bounds:is-an-error" "iss332.htm")
     ("subsetting-position:none" "iss333.htm")
     ("subtypep-environment:add-arg" "iss334.htm")
     ("subtypep-too-vague:clarify-more" "iss335.htm")
     ("sxhash-definition:similar-for-sxhash" "iss336.htm")
     ("symbol-macrolet-declare:allow" "iss337.htm")
     ("symbol-macrolet-semantics:special-form" "iss338.htm")
     ("symbol-macrolet-type-declaration:no" "iss339.htm")
     ("symbol-macros-and-proclaimed-specials:signals-an-error" "iss340.htm")
     ("symbol-print-escape-behavior:clarify" "iss341.htm")
     ("syntactic-environment-access:retracted-mar91" "iss342.htm")
     ("tagbody-tag-expansion:no" "iss343.htm")
     ("tailp-nil:t" "iss344.htm")
     ("test-not-if-not:flush-all" "iss345.htm")
     ("the-ambiguity:for-declaration" "iss346.htm")
     ("the-values:return-number-received" "iss347.htm")
     ("time-zone-non-integer:allow" "iss348.htm")
     ("type-declaration-abbreviation:allow-all" "iss349.htm")
     ("type-of-and-predefined-classes:type-of-handles-floats" "iss350.htm")
     ("type-of-and-predefined-classes:unify-and-extend" "iss351.htm")
     ("type-of-underconstrained:add-constraints" "iss352.htm")
     ("type-specifier-abbreviation:x3j13-jun90-guess" "iss353.htm")
     ("undefined-variables-and-functions:compromise" "iss354.htm")
     ("uninitialized-elements:consequences-undefined" "iss355.htm")
     ("unread-char-after-peek-char:dont-allow" "iss356.htm")
     ("unsolicited-messages:not-to-system-user-streams" "iss357.htm")
     ("variable-list-asymmetry:symmetrize" "iss358.htm")
     ("with-added-methods:delete" "iss359.htm")
     ("with-compilation-unit:new-macro" "iss360.htm")
     ("with-open-file-does-not-exist:stream-is-nil" "iss361.htm")
     ("with-open-file-setq:explicitly-vague" "iss362.htm")
     ("with-open-file-stream-extent:dynamic-extent" "iss363.htm")
     ("with-output-to-string-append-style:vector-push-extend" "iss364.htm")
     ("with-standard-io-syntax-readtable:x3j13-mar-91" "iss365.htm"))))

(defun common-lisp-issuex (issue-name)
  (let ((entry (gethash (downcase issue-name)
			common-lisp-hyperspec--issuex-symbols)))
    (concat common-lisp-hyperspec-root "Issues/" entry)))

(defun common-lisp-special-operator (name)
  (format "%sBody/s_%s.htm" common-lisp-hyperspec-root name))

;;; Added the following just to provide a common entry point according
;;; to the various 'hyperspec' implementations.
;;;
;;; 19990820 Marco Antoniotti

(defalias 'hyperspec-lookup 'common-lisp-hyperspec)
(defalias 'hyperspec-lookup-reader-macro
  'common-lisp-hyperspec-lookup-reader-macro)
(defalias 'hyperspec-lookup-format 'common-lisp-hyperspec-format)

(provide 'hyperspec)

;;; hyperspec.el ends here
